'use client';

import { useState, useEffect } from 'react';
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

export default function SavingsGoalCalculator() {
  const [goalAmount, setGoalAmount] = useState<string>('100000');
  const [currentSavings, setCurrentSavings] = useState<string>('10000');
  const [monthlyContribution, setMonthlyContribution] = useState<string>('2000');
  const [interestRate, setInterestRate] = useState<string>('6');
  const [timeframe, setTimeframe] = useState<string>('36');
  const [calculationMode, setCalculationMode] = useState<'time' | 'payment'>('time');
  
  const [monthsNeeded, setMonthsNeeded] = useState<number>(0);
  const [requiredMonthlyPayment, setRequiredMonthlyPayment] = useState<number>(0);
  const [totalContributions, setTotalContributions] = useState<number>(0);
  const [totalInterest, setTotalInterest] = useState<number>(0);
  const [finalAmount, setFinalAmount] = useState<number>(0);
  const [breakdown, setBreakdown] = useState<any[]>([]);

  useEffect(() => {
    if (calculationMode === 'time') {
      calculateTimeToGoal();
    } else {
      calculateRequiredPayment();
    }
  }, [goalAmount, currentSavings, monthlyContribution, interestRate, timeframe, calculationMode]);

  const calculateTimeToGoal = () => {
    const goal = parseFloat(goalAmount) || 0;
    const current = parseFloat(currentSavings) || 0;
    const payment = parseFloat(monthlyContribution) || 0;
    const rate = parseFloat(interestRate) / 100 / 12 || 0;

    if (goal <= current) {
      setMonthsNeeded(0);
      setTotalContributions(0);
      setTotalInterest(0);
      setFinalAmount(current);
      setBreakdown([]);
      return;
    }

    let balance = current;
    let months = 0;
    let totalPaid = 0;
    const schedule = [];
    const maxMonths = 600; // 50 years max

    while (balance < goal && months < maxMonths) {
      months++;
      const interest = balance * rate;
      balance += payment + interest;
      totalPaid += payment;

      if (months <= 12 || months % 6 === 0 || balance >= goal) {
        schedule.push({
          month: months,
          contribution: payment,
          interest: interest,
          balance: Math.min(balance, goal),
          totalContributions: totalPaid,
          totalInterest: balance - current - totalPaid
        });
      }

      if (balance >= goal) break;
    }

    const totalInt = balance - current - totalPaid;
    
    setMonthsNeeded(months);
    setTotalContributions(totalPaid);
    setTotalInterest(totalInt);
    setFinalAmount(balance);
    setBreakdown(schedule);
  };

  const calculateRequiredPayment = () => {
    const goal = parseFloat(goalAmount) || 0;
    const current = parseFloat(currentSavings) || 0;
    const months = parseInt(timeframe) || 1;
    const rate = parseFloat(interestRate) / 100 / 12 || 0;

    if (goal <= current) {
      setRequiredMonthlyPayment(0);
      setTotalContributions(0);
      setTotalInterest(0);
      setFinalAmount(current);
      setBreakdown([]);
      return;
    }

    // Calculate required monthly payment using future value of annuity formula
    // FV = PV(1+r)^n + PMT * [((1+r)^n - 1) / r]
    // Solving for PMT: PMT = (FV - PV(1+r)^n) / [((1+r)^n - 1) / r]
    
    const futureValueOfCurrent = current * Math.pow(1 + rate, months);
    const remaining = goal - futureValueOfCurrent;
    
    let payment = 0;
    if (rate > 0) {
      payment = remaining / (((Math.pow(1 + rate, months) - 1) / rate));
    } else {
      payment = remaining / months;
    }

    // Generate breakdown
    let balance = current;
    let totalPaid = 0;
    const schedule = [];

    for (let m = 1; m <= months; m++) {
      const interest = balance * rate;
      balance += payment + interest;
      totalPaid += payment;

      if (m <= 12 || m % 6 === 0 || m === months) {
        schedule.push({
          month: m,
          contribution: payment,
          interest: interest,
          balance: balance,
          totalContributions: totalPaid,
          totalInterest: balance - current - totalPaid
        });
      }
    }

    const totalInt = balance - current - totalPaid;

    setRequiredMonthlyPayment(payment);
    setTotalContributions(totalPaid);
    setTotalInterest(totalInt);
    setFinalAmount(balance);
    setBreakdown(schedule);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const yearsNeeded = monthsNeeded / 12;
  const progressPercentage = parseFloat(goalAmount) > 0 
    ? ((parseFloat(currentSavings) || 0) / parseFloat(goalAmount)) * 100 
    : 0;

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Savings Goal Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate how much you need to save regularly to reach your financial goals in South Africa.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        {/* Mode Toggle */}
        <div className="flex justify-center mb-6">
          <div className="inline-flex rounded-lg border border-gray-300 p-1">
            <button
              type="button"
              onClick={() => setCalculationMode('time')}
              className={`px-6 py-2 rounded-md text-sm font-medium transition-colors ${
                calculationMode === 'time'
                  ? 'bg-blue-600 text-white'
                  : 'text-gray-700 hover:bg-gray-100'
              }`}
            >
              Calculate Time Needed
            </button>
            <button
              type="button"
              onClick={() => setCalculationMode('payment')}
              className={`px-6 py-2 rounded-md text-sm font-medium transition-colors ${
                calculationMode === 'payment'
                  ? 'bg-blue-600 text-white'
                  : 'text-gray-700 hover:bg-gray-100'
              }`}
            >
              Calculate Required Payment
            </button>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="md:col-span-2">
                <label htmlFor="goal-amount" className="block text-sm font-medium text-gray-700 mb-1">
                  Savings Goal (R)
                </label>
                <input
                  type="text"
                  id="goal-amount"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                  value={goalAmount}
                  onChange={(e) => setGoalAmount(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="100000"
                />
              </div>

              <div>
                <label htmlFor="current-savings" className="block text-sm font-medium text-gray-700 mb-1">
                  Current Savings (R)
                </label>
                <input
                  type="text"
                  id="current-savings"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={currentSavings}
                  onChange={(e) => setCurrentSavings(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="10000"
                />
              </div>

              {calculationMode === 'time' ? (
                <div>
                  <label htmlFor="monthly-contribution" className="block text-sm font-medium text-gray-700 mb-1">
                    Monthly Contribution (R)
                  </label>
                  <input
                    type="text"
                    id="monthly-contribution"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={monthlyContribution}
                    onChange={(e) => setMonthlyContribution(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="2000"
                  />
                </div>
              ) : (
                <div>
                  <label htmlFor="timeframe" className="block text-sm font-medium text-gray-700 mb-1">
                    Timeframe (Months)
                  </label>
                  <input
                    type="text"
                    id="timeframe"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={timeframe}
                    onChange={(e) => setTimeframe(e.target.value.replace(/[^0-9]/g, ''))}
                    placeholder="36"
                  />
                </div>
              )}

              <div className="md:col-span-2">
                <label htmlFor="interest-rate" className="block text-sm font-medium text-gray-700 mb-1">
                  Expected Annual Return (%)
                </label>
                <input
                  type="text"
                  id="interest-rate"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={interestRate}
                  onChange={(e) => setInterestRate(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="6"
                />
                <p className="mt-1 text-xs text-gray-500">Savings account: 4-6%, Investments: 8-12%</p>
              </div>
            </div>

            {/* Progress Bar */}
            <div className="border-t pt-4">
              <div className="flex justify-between text-sm mb-2">
                <span className="text-gray-600">Current Progress</span>
                <span className="font-medium text-gray-900">{progressPercentage.toFixed(1)}%</span>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-4">
                <div 
                  className="bg-blue-600 h-4 rounded-full transition-all duration-300"
                  style={{ width: `${Math.min(progressPercentage, 100)}%` }}
                ></div>
              </div>
              <div className="flex justify-between text-xs text-gray-500 mt-1">
                <span>{formatCurrency(parseFloat(currentSavings) || 0)}</span>
                <span>{formatCurrency(parseFloat(goalAmount) || 0)}</span>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-green-50 to-emerald-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Goal Summary</h3>
              
              <div className="space-y-3">
                {calculationMode === 'time' ? (
                  <>
                    <div>
                      <p className="text-xs text-gray-600">Time to Reach Goal</p>
                      <p className="text-2xl font-bold text-green-600">
                        {monthsNeeded > 0 ? `${Math.floor(yearsNeeded)} yrs ${monthsNeeded % 12} mos` : 'Goal Reached!'}
                      </p>
                    </div>
                    <div className="border-t border-emerald-200 pt-3">
                      <p className="text-xs text-gray-600">Monthly Contribution</p>
                      <p className="text-lg font-semibold text-gray-900">{formatCurrency(parseFloat(monthlyContribution) || 0)}</p>
                    </div>
                  </>
                ) : (
                  <>
                    <div>
                      <p className="text-xs text-gray-600">Required Monthly Payment</p>
                      <p className="text-2xl font-bold text-green-600">{formatCurrency(requiredMonthlyPayment)}</p>
                    </div>
                    <div className="border-t border-emerald-200 pt-3">
                      <p className="text-xs text-gray-600">Timeframe</p>
                      <p className="text-lg font-semibold text-gray-900">
                        {Math.floor(parseInt(timeframe) / 12)} yrs {parseInt(timeframe) % 12} mos
                      </p>
                    </div>
                  </>
                )}

                <div className="border-t border-emerald-200 pt-3">
                  <p className="text-xs text-gray-600">Total Contributions</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(totalContributions)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Interest Earned</p>
                  <p className="text-sm font-medium text-green-600">{formatCurrency(totalInterest)}</p>
                </div>

                <div className="border-t border-emerald-200 pt-3">
                  <p className="text-xs text-gray-600">Final Amount</p>
                  <p className="text-lg font-bold text-gray-900">{formatCurrency(finalAmount)}</p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <h4 className="font-semibold text-gray-900 mb-2">Breakdown</h4>
              <ul className="space-y-1 text-gray-600 text-xs">
                <li>• Starting: {formatCurrency(parseFloat(currentSavings) || 0)}</li>
                <li>• You contribute: {formatCurrency(totalContributions)}</li>
                <li>• Interest adds: {formatCurrency(totalInterest)}</li>
                <li>• Final total: {formatCurrency(finalAmount)}</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Savings Schedule */}
        {breakdown.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Savings Schedule</h3>
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Month</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Contribution</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Interest</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Balance</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Total Saved</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Total Interest</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {breakdown.map((row, index) => (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="px-4 py-3 text-sm text-gray-900">{row.month}</td>
                      <td className="px-4 py-3 text-sm text-right text-blue-600">{formatCurrency(row.contribution)}</td>
                      <td className="px-4 py-3 text-sm text-right text-green-600">{formatCurrency(row.interest)}</td>
                      <td className="px-4 py-3 text-sm text-right font-medium text-gray-900">{formatCurrency(row.balance)}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{formatCurrency(row.totalContributions)}</td>
                      <td className="px-4 py-3 text-sm text-right text-green-600">{formatCurrency(row.totalInterest)}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
      </div>

      <InContentAd />

      {/* Savings Growth Chart */}
      {breakdown.length > 0 && (
        <div className="bg-white rounded-xl shadow-md p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Savings Growth Over Time</h3>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={breakdown}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="month" label={{ value: 'Month', position: 'insideBottom', offset: -5 }} />
              <YAxis tickFormatter={(value) => `R${(value / 1000).toFixed(0)}k`} />
              <Tooltip formatter={(value: number) => formatCurrency(value)} />
              <Legend />
              <Line type="monotone" dataKey="balance" stroke="#3b82f6" strokeWidth={2} name="Total Balance" />
              <Line type="monotone" dataKey="totalContributions" stroke="#10b981" strokeWidth={2} name="Contributions" />
            </LineChart>
          </ResponsiveContainer>
        </div>
      )}

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="savings-goal" category="Savings" />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Achieving Your Savings Goals in South Africa</h2>
        <p>
          Whether you're saving for a house deposit, emergency fund, car, holiday, or retirement, having a clear 
          savings goal and plan is essential. This calculator helps you determine how much to save and how long it will take.
        </p>

        <h2>How to Use the Savings Goal Calculator</h2>
        <ol>
          <li>Choose your calculation mode (time needed or required payment)</li>
          <li>Enter your savings goal amount</li>
          <li>Input your current savings</li>
          <li>Set your monthly contribution or timeframe</li>
          <li>Enter expected annual return (interest rate)</li>
          <li>View your personalized savings plan</li>
        </ol>

        <h2>Common Savings Goals in South Africa</h2>
        <ul>
          <li><strong>Emergency Fund:</strong> 3-6 months of expenses (R30,000-R100,000)</li>
          <li><strong>House Deposit:</strong> 10-20% of property value (R150,000-R500,000)</li>
          <li><strong>Car Purchase:</strong> Full amount or deposit (R50,000-R300,000)</li>
          <li><strong>Holiday/Travel:</strong> Domestic or international trip (R20,000-R100,000)</li>
          <li><strong>Education:</strong> University fees or courses (R50,000-R200,000)</li>
          <li><strong>Wedding:</strong> Average SA wedding (R80,000-R200,000)</li>
          <li><strong>Retirement:</strong> Long-term wealth building (R1M-R10M+)</li>
        </ul>

        <h2>Where to Save Your Money</h2>
        
        <h3>Short-Term Goals (0-2 years)</h3>
        <ul>
          <li><strong>Savings Account:</strong> 4-6% interest, instant access</li>
          <li><strong>Money Market Account:</strong> 5-7% interest, limited withdrawals</li>
          <li><strong>Fixed Deposit:</strong> 6-8% interest, locked for term</li>
          <li><strong>Tax-Free Savings (TFSA):</strong> Tax-free growth, R36,000/year limit</li>
        </ul>

        <h3>Medium-Term Goals (2-5 years)</h3>
        <ul>
          <li><strong>Unit Trusts:</strong> 8-12% returns, moderate risk</li>
          <li><strong>ETFs:</strong> Market returns, low fees</li>
          <li><strong>TFSA (invested):</strong> Tax-free investment growth</li>
          <li><strong>Balanced Funds:</strong> Mix of shares and bonds</li>
        </ul>

        <h3>Long-Term Goals (5+ years)</h3>
        <ul>
          <li><strong>Retirement Annuity:</strong> Tax benefits, long-term growth</li>
          <li><strong>Share Portfolio:</strong> 10-15% historical returns</li>
          <li><strong>Property Investment:</strong> Rental income + capital growth</li>
          <li><strong>Offshore Investments:</strong> Diversification, Rand hedge</li>
        </ul>

        <h2>The Power of Compound Interest</h2>
        <p>
          Compound interest is when you earn interest on your interest. Over time, this creates exponential growth:
        </p>
        <ul>
          <li>Save R2,000/month at 6% for 10 years = R330,000 (R90,000 interest)</li>
          <li>Save R2,000/month at 6% for 20 years = R920,000 (R440,000 interest)</li>
          <li>Save R2,000/month at 6% for 30 years = R2,000,000 (R1,280,000 interest)</li>
        </ul>

        <h2>Tips for Reaching Your Savings Goals</h2>
        <ul>
          <li><strong>Automate Savings:</strong> Set up automatic transfers on payday</li>
          <li><strong>Pay Yourself First:</strong> Save before spending</li>
          <li><strong>Start Small:</strong> Even R500/month adds up over time</li>
          <li><strong>Increase Gradually:</strong> Boost savings with salary increases</li>
          <li><strong>Track Progress:</strong> Review monthly to stay motivated</li>
          <li><strong>Avoid Withdrawals:</strong> Let compound interest work</li>
          <li><strong>Use Windfalls:</strong> Add bonuses and tax refunds to savings</li>
        </ul>

        <h2>Tax-Free Savings Accounts (TFSA)</h2>
        <p>
          South Africans can save up to R36,000 per year (R500,000 lifetime) in a TFSA:
        </p>
        <ul>
          <li>No tax on interest, dividends, or capital gains</li>
          <li>Can be in cash or investments</li>
          <li>Ideal for medium to long-term goals</li>
          <li>Penalties for exceeding limits</li>
        </ul>

        <h2>Common Savings Mistakes to Avoid</h2>
        <ul>
          <li><strong>No Clear Goal:</strong> Vague goals lead to poor discipline</li>
          <li><strong>Unrealistic Targets:</strong> Set achievable amounts and timeframes</li>
          <li><strong>Not Adjusting for Inflation:</strong> Your goal should grow with inflation</li>
          <li><strong>Keeping Cash:</strong> Inflation erodes cash value, invest appropriately</li>
          <li><strong>Dipping In:</strong> Withdrawals derail progress and compound interest</li>
          <li><strong>Ignoring Fees:</strong> High fees reduce returns significantly</li>
        </ul>

        <h2>Emergency Fund Priority</h2>
        <p>
          Before saving for other goals, build an emergency fund:
        </p>
        <ul>
          <li>Start with R10,000 mini emergency fund</li>
          <li>Build to 3-6 months of expenses</li>
          <li>Keep in accessible savings account</li>
          <li>Only use for true emergencies</li>
          <li>Replenish immediately after use</li>
        </ul>

        <div className="bg-green-50 border-l-4 border-green-400 p-4 my-6">
          <p className="text-sm">
            <strong>Success Tip:</strong> The best time to start saving was yesterday. The second best time is today. 
            Even small amounts grow significantly over time thanks to compound interest. Start now, stay consistent, 
            and watch your savings grow!
          </p>
        </div>
      </div>
    </div>
  );
}
