'use client';

import { useState, useEffect, useMemo } from 'react';
import { LineChart, Line, AreaChart, Area, PieChart, Pie, Cell, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

export default function RetirementCalculator() {
  // Input states
  const [currentAge, setCurrentAge] = useState<string>('30');
  const [retirementAge, setRetirementAge] = useState<string>('65');
  const [currentSavings, setCurrentSavings] = useState<string>('50000');
  const [monthlyContribution, setMonthlyContribution] = useState<string>('2000');
  const [annualReturn, setAnnualReturn] = useState<string>('7');
  const [inflationRate, setInflationRate] = useState<string>('3');
  const [desiredMonthlyIncome, setDesiredMonthlyIncome] = useState<string>('20000');
  const [yearsInRetirement, setYearsInRetirement] = useState<string>('25');

  // Results
  const [totalSavings, setTotalSavings] = useState<number>(0);
  const [totalContributions, setTotalContributions] = useState<number>(0);
  const [totalGrowth, setTotalGrowth] = useState<number>(0);
  const [monthlyRetirementIncome, setMonthlyRetirementIncome] = useState<number>(0);
  const [savingsNeeded, setSavingsNeeded] = useState<number>(0);
  const [shortfall, setShortfall] = useState<number>(0);
  const [yearlyBreakdown, setYearlyBreakdown] = useState<any[]>([]);

  useEffect(() => {
    calculateRetirement();
  }, [
    currentAge,
    retirementAge,
    currentSavings,
    monthlyContribution,
    annualReturn,
    inflationRate,
    desiredMonthlyIncome,
    yearsInRetirement,
  ]);

  const calculateRetirement = () => {
    const age = parseInt(currentAge);
    const retAge = parseInt(retirementAge);
    const savings = parseFloat(currentSavings) || 0;
    const monthly = parseFloat(monthlyContribution) || 0;
    const returnRate = parseFloat(annualReturn) / 100 || 0;
    const inflation = parseFloat(inflationRate) / 100 || 0;
    const desiredIncome = parseFloat(desiredMonthlyIncome) || 0;
    const retYears = parseInt(yearsInRetirement) || 25;

    if (age >= retAge || retAge <= 0 || age <= 0) {
      resetResults();
      return;
    }

    const yearsToRetirement = retAge - age;
    const monthlyRate = returnRate / 12;
    const totalMonths = yearsToRetirement * 12;

    // Calculate future value of current savings
    const futureValueCurrentSavings = savings * Math.pow(1 + returnRate, yearsToRetirement);

    // Calculate future value of monthly contributions (annuity)
    let futureValueContributions = 0;
    if (monthlyRate > 0) {
      futureValueContributions = monthly * ((Math.pow(1 + monthlyRate, totalMonths) - 1) / monthlyRate);
    } else {
      futureValueContributions = monthly * totalMonths;
    }

    const totalAtRetirement = futureValueCurrentSavings + futureValueContributions;
    const totalContrib = monthly * totalMonths;
    const totalGain = totalAtRetirement - savings - totalContrib;

    // Calculate how much is needed for desired retirement income
    // Adjust desired income for inflation
    const inflationAdjustedIncome = desiredIncome * Math.pow(1 + inflation, yearsToRetirement);
    
    // Calculate present value of retirement income needed (annuity)
    const monthlyRateRetirement = returnRate / 12;
    const totalRetirementMonths = retYears * 12;
    
    let neededSavings = 0;
    if (monthlyRateRetirement > 0) {
      neededSavings = inflationAdjustedIncome * ((1 - Math.pow(1 + monthlyRateRetirement, -totalRetirementMonths)) / monthlyRateRetirement);
    } else {
      neededSavings = inflationAdjustedIncome * totalRetirementMonths;
    }

    // Calculate monthly income from total savings
    let monthlyIncome = 0;
    if (monthlyRateRetirement > 0) {
      monthlyIncome = totalAtRetirement * (monthlyRateRetirement / (1 - Math.pow(1 + monthlyRateRetirement, -totalRetirementMonths)));
    } else {
      monthlyIncome = totalAtRetirement / totalRetirementMonths;
    }

    const gap = neededSavings - totalAtRetirement;

    // Generate yearly breakdown
    const breakdown = [];
    let balance = savings;
    const currentYear = new Date().getFullYear();
    
    for (let year = 0; year <= yearsToRetirement; year++) {
      const yearContributions = monthly * 12;
      const yearGrowth = balance * returnRate;
      balance = balance + yearContributions + yearGrowth;
      
      breakdown.push({
        year: currentYear + year,
        age: age + year,
        startBalance: year === 0 ? savings : breakdown[year - 1].endBalance,
        contributions: yearContributions,
        growth: yearGrowth,
        endBalance: balance,
      });
    }

    setTotalSavings(totalAtRetirement);
    setTotalContributions(totalContrib);
    setTotalGrowth(totalGain);
    setMonthlyRetirementIncome(monthlyIncome);
    setSavingsNeeded(neededSavings);
    setShortfall(gap);
    setYearlyBreakdown(breakdown);
  };

  const resetResults = () => {
    setTotalSavings(0);
    setTotalContributions(0);
    setTotalGrowth(0);
    setMonthlyRetirementIncome(0);
    setSavingsNeeded(0);
    setShortfall(0);
    setYearlyBreakdown([]);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const yearsToRetirement = parseInt(retirementAge) - parseInt(currentAge);
  const isOnTrack = shortfall <= 0;

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Retirement Calculator</h1>
        <p className="mt-2 text-gray-600">
          Plan your retirement savings and see if you're on track to meet your retirement goals.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Current Age */}
              <div>
                <label htmlFor="current-age" className="block text-sm font-medium text-gray-700 mb-1">
                  Current Age
                </label>
                <input
                  type="number"
                  id="current-age"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={currentAge}
                  onChange={(e) => setCurrentAge(e.target.value)}
                  min="18"
                  max="100"
                />
              </div>

              {/* Retirement Age */}
              <div>
                <label htmlFor="retirement-age" className="block text-sm font-medium text-gray-700 mb-1">
                  Retirement Age
                </label>
                <input
                  type="number"
                  id="retirement-age"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={retirementAge}
                  onChange={(e) => setRetirementAge(e.target.value)}
                  min="50"
                  max="100"
                />
              </div>

              {/* Current Savings */}
              <div>
                <label htmlFor="current-savings" className="block text-sm font-medium text-gray-700 mb-1">
                  Current Retirement Savings (R)
                </label>
                <input
                  type="text"
                  id="current-savings"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={currentSavings}
                  onChange={(e) => setCurrentSavings(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="50000"
                />
              </div>

              {/* Monthly Contribution */}
              <div>
                <label htmlFor="monthly-contribution" className="block text-sm font-medium text-gray-700 mb-1">
                  Monthly Contribution (R)
                </label>
                <input
                  type="text"
                  id="monthly-contribution"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={monthlyContribution}
                  onChange={(e) => setMonthlyContribution(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="2000"
                />
              </div>

              {/* Expected Annual Return */}
              <div>
                <label htmlFor="annual-return" className="block text-sm font-medium text-gray-700 mb-1">
                  Expected Annual Return (%)
                </label>
                <input
                  type="text"
                  id="annual-return"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={annualReturn}
                  onChange={(e) => setAnnualReturn(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="7"
                />
                <p className="mt-1 text-xs text-gray-500">Typical range: 5-10%</p>
              </div>

              {/* Inflation Rate */}
              <div>
                <label htmlFor="inflation-rate" className="block text-sm font-medium text-gray-700 mb-1">
                  Expected Inflation Rate (%)
                </label>
                <input
                  type="text"
                  id="inflation-rate"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={inflationRate}
                  onChange={(e) => setInflationRate(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="3"
                />
                <p className="mt-1 text-xs text-gray-500">SA average: 3-6%</p>
              </div>

              {/* Desired Monthly Income */}
              <div>
                <label htmlFor="desired-income" className="block text-sm font-medium text-gray-700 mb-1">
                  Desired Monthly Income in Retirement (R)
                </label>
                <input
                  type="text"
                  id="desired-income"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={desiredMonthlyIncome}
                  onChange={(e) => setDesiredMonthlyIncome(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="20000"
                />
                <p className="mt-1 text-xs text-gray-500">In today's Rands</p>
              </div>

              {/* Years in Retirement */}
              <div>
                <label htmlFor="years-retirement" className="block text-sm font-medium text-gray-700 mb-1">
                  Expected Years in Retirement
                </label>
                <input
                  type="number"
                  id="years-retirement"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={yearsInRetirement}
                  onChange={(e) => setYearsInRetirement(e.target.value)}
                  min="10"
                  max="50"
                />
                <p className="mt-1 text-xs text-gray-500">Typical: 20-30 years</p>
              </div>
            </div>
          </div>

          {/* Right Column - Summary */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Retirement Summary</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Years to Retirement</p>
                  <p className="text-2xl font-bold text-blue-600">{yearsToRetirement > 0 ? yearsToRetirement : 0}</p>
                </div>

                <div className="border-t border-blue-200 pt-3">
                  <p className="text-xs text-gray-600">Total at Retirement</p>
                  <p className="text-xl font-bold text-gray-900">{formatCurrency(totalSavings)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Monthly Income</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(monthlyRetirementIncome)}</p>
                </div>

                <div className="border-t border-blue-200 pt-3">
                  <p className="text-xs text-gray-600">Savings Needed</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(savingsNeeded)}</p>
                </div>

                <div className={`rounded-lg p-3 ${isOnTrack ? 'bg-green-100' : 'bg-red-100'}`}>
                  <p className="text-xs font-medium text-gray-700">Status</p>
                  <p className={`text-sm font-bold ${isOnTrack ? 'text-green-700' : 'text-red-700'}`}>
                    {isOnTrack ? '✓ On Track!' : `Shortfall: ${formatCurrency(Math.abs(shortfall))}`}
                  </p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <h4 className="font-semibold text-gray-900 mb-2">Breakdown</h4>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-gray-600">Your Contributions:</span>
                  <span className="font-medium">{formatCurrency(totalContributions)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Investment Growth:</span>
                  <span className="font-medium text-green-600">{formatCurrency(totalGrowth)}</span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Yearly Breakdown Table */}
        {/* Growth Chart */}
        {yearlyBreakdown.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Savings Growth Projection</h3>
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <ResponsiveContainer width="100%" height={300}>
                <AreaChart data={yearlyBreakdown.slice(0, Math.min(yearlyBreakdown.length, 20))}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="year" />
                  <YAxis tickFormatter={(value) => `R${(value / 1000).toFixed(0)}k`} />
                  <Tooltip formatter={(value: number) => formatCurrency(value)} />
                  <Legend />
                  <Area type="monotone" dataKey="endBalance" stackId="1" stroke="#3b82f6" fill="#3b82f6" name="Total Balance" />
                </AreaChart>
              </ResponsiveContainer>
            </div>
          </div>
        )}

        {/* Contributions vs Growth Pie Chart */}
        {totalContributions > 0 && totalGrowth > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Savings Breakdown</h3>
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <ResponsiveContainer width="100%" height={300}>
                <PieChart>
                  <Pie
                    data={[
                      { name: 'Your Contributions', value: parseFloat(currentSavings) + totalContributions, fill: '#3b82f6' },
                      { name: 'Investment Growth', value: totalGrowth, fill: '#10b981' },
                    ]}
                    cx="50%"
                    cy="50%"
                    labelLine={false}
                    label={({ name, percent }: any) => `${name}: ${(percent * 100).toFixed(0)}%`}
                    outerRadius={100}
                    dataKey="value"
                  >
                  </Pie>
                  <Tooltip formatter={(value: number) => formatCurrency(value)} />
                </PieChart>
              </ResponsiveContainer>
            </div>
          </div>
        )}

        {yearlyBreakdown.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Year-by-Year Projection</h3>
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Year</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Age</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Start Balance</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Contributions</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Growth</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">End Balance</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {yearlyBreakdown.slice(0, 10).map((row, index) => (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="px-4 py-3 text-sm text-gray-900">{row.year}</td>
                      <td className="px-4 py-3 text-sm text-gray-900">{row.age}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{formatCurrency(row.startBalance)}</td>
                      <td className="px-4 py-3 text-sm text-right text-blue-600">{formatCurrency(row.contributions)}</td>
                      <td className="px-4 py-3 text-sm text-right text-green-600">{formatCurrency(row.growth)}</td>
                      <td className="px-4 py-3 text-sm text-right font-medium text-gray-900">{formatCurrency(row.endBalance)}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
              {yearlyBreakdown.length > 10 && (
                <p className="text-sm text-gray-500 mt-2 text-center">Showing first 10 years of {yearlyBreakdown.length} years</p>
              )}
            </div>
          </div>
        )}
      </div>

      <InContentAd />

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="retirement-calculator" category="Retirement" />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>How to Use the Retirement Calculator</h2>
        <ol>
          <li>Enter your current age and desired retirement age</li>
          <li>Input your current retirement savings amount</li>
          <li>Specify how much you plan to contribute monthly</li>
          <li>Set your expected annual investment return (typically 5-10%)</li>
          <li>Enter the expected inflation rate (SA average is 3-6%)</li>
          <li>Specify your desired monthly income in retirement (in today's Rands)</li>
          <li>Set how many years you expect to be in retirement</li>
        </ol>

        <h2>Understanding Your Results</h2>
        <p>
          The calculator shows you whether you're on track to meet your retirement goals. If there's a shortfall, 
          consider increasing your monthly contributions or adjusting your retirement expectations.
        </p>

        <h2>Retirement Planning Tips for South Africans</h2>
        <ul>
          <li><strong>Start Early:</strong> The power of compound interest means starting early makes a huge difference</li>
          <li><strong>Maximize Tax Benefits:</strong> Contribute to retirement annuities (RAs) for tax deductions up to 27.5% of income</li>
          <li><strong>Diversify:</strong> Don't put all your retirement savings in one investment type</li>
          <li><strong>Consider Inflation:</strong> Your retirement income needs to keep pace with inflation</li>
          <li><strong>Review Regularly:</strong> Check your retirement plan at least annually and adjust as needed</li>
          <li><strong>Employer Contributions:</strong> Take full advantage of employer pension or provident fund matching</li>
        </ul>

        <h2>South African Retirement Vehicles</h2>
        <ul>
          <li><strong>Retirement Annuity (RA):</strong> Tax-efficient retirement savings with contributions deductible up to 27.5% of income</li>
          <li><strong>Pension Fund:</strong> Employer-sponsored retirement fund</li>
          <li><strong>Provident Fund:</strong> Similar to pension fund with different payout options</li>
          <li><strong>Tax-Free Savings Account (TFSA):</strong> R36,000 annual limit, tax-free growth</li>
          <li><strong>Unit Trusts:</strong> Investment funds for long-term growth</li>
        </ul>

        <h2>Important Considerations</h2>
        <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4">
          <p className="text-sm">
            <strong>Disclaimer:</strong> This calculator provides estimates based on the assumptions you enter. 
            Actual investment returns will vary, and past performance doesn't guarantee future results. 
            Consult with a qualified financial advisor for personalized retirement planning advice.
          </p>
        </div>
      </div>
    </div>
  );
}
