import { calculators } from '@/data/calculators';
import Link from 'next/link';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export const metadata = {
  title: 'All Financial Calculators - SA PLUK',
  description: 'Browse our complete collection of free online financial calculators for South Africa. Tools for loans, mortgages, investments, VAT, currency conversion, and more.',
};

export default function CalculatorsPage() {
  // Group calculators by category
  const calculatorsByCategory = calculators.reduce((acc, calculator) => {
    if (!acc[calculator.category]) {
      acc[calculator.category] = [];
    }
    acc[calculator.category].push(calculator);
    return acc;
  }, {} as Record<string, typeof calculators>);

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div className="bg-gradient-to-br from-gray-50 via-blue-50 to-purple-50 py-20 rounded-lg shadow-lg border-2 border-blue-100">
        <div className="container mx-auto px-4">
          <div className="inline-block mb-4 px-4 py-2 bg-blue-600 text-white text-xs font-bold tracking-widest uppercase shadow-lg rounded-full">
            Our Tools
          </div>
          <h1 className="text-5xl md:text-6xl font-black mb-6 text-gray-900">Financial Calculators</h1>
          <div className="w-32 h-2 bg-gradient-to-r from-cyan-400 via-cyan-500 to-cyan-400 mx-auto mb-6 rounded-full"></div>
          <p className="text-lg max-w-3xl mx-auto text-gray-700">
            Free online tools to help you make informed financial decisions. Calculate loans, home bonds, investments, VAT, and more.
          </p>
        </div>
      </div>

      {/* Top Banner Ad */}
      <LeaderboardAd />

      {/* Search and Filter (Placeholder for future implementation) */}
      <div className="bg-white rounded-lg shadow p-4">
        <div className="flex flex-col md:flex-row gap-4">
          <input
            type="text"
            placeholder="Search calculators..."
            className="flex-grow p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            disabled
          />
          <select 
            className="p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-gray-700"
            disabled
          >
            <option value="">All Categories</option>
            {Object.keys(calculatorsByCategory).map(category => (
              <option key={category} value={category}>{category}</option>
            ))}
          </select>
          <button 
            className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition duration-300"
            disabled
          >
            Search
          </button>
        </div>
      </div>

      {/* In-Content Ad */}
      <InContentAd />

      {/* Featured Calculators */}
      <section>
        <h2 className="text-2xl font-bold mb-6 text-gray-800">Featured Calculators</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-12">
          {calculators.slice(0, 6).map((calculator) => (
            <CalculatorCard key={calculator.id} calculator={calculator} />
          ))}
        </div>
      </section>

      {/* All Calculators by Category */}
      <div className="space-y-12">
        {Object.entries(calculatorsByCategory).map(([category, categoryCalculators]) => (
          <section key={category}>
            <h2 className="text-2xl font-bold mb-6 text-gray-800 border-b pb-2">{category}</h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-12">
              {categoryCalculators.map((calculator) => (
                <CalculatorCard key={calculator.id} calculator={calculator} />
              ))}
            </div>
          </section>
        ))}
      </div>

      {/* Bottom CTA */}
      <section className="bg-blue-50 rounded-lg p-8 text-center">
        <h2 className="text-2xl font-bold mb-4">Can't Find What You're Looking For?</h2>
        <p className="text-gray-700 mb-6 max-w-2xl mx-auto">
          We're constantly adding new calculators. Let us know what financial tools you'd like to see!
        </p>
        <Link 
          href="/contact" 
          className="inline-block bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition duration-300"
        >
          Suggest a Calculator
        </Link>
      </section>
    </div>
  );
}

function CalculatorCard({ calculator }: { calculator: typeof calculators[0] }) {
  return (
    <Link 
      href={calculator.path}
      className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-xl transition-shadow duration-300 border border-gray-200 flex flex-col h-full"
    >
      <div className="p-6 flex-grow">
        <div className="text-4xl mb-4">{calculator.icon}</div>
        <h3 className="text-xl font-semibold mb-2 text-gray-800">{calculator.title}</h3>
        <p className="text-gray-600 mb-4">{calculator.description}</p>
      </div>
      <div className="px-6 py-3 bg-gray-50 border-t border-gray-200">
        <div className="flex flex-wrap gap-2">
          <span className="inline-block bg-blue-100 text-blue-800 text-xs px-3 py-1 rounded-full">
            {calculator.category}
          </span>
          {calculator.tags.slice(0, 2).map(tag => (
            <span key={tag} className="inline-block bg-gray-100 text-gray-600 text-xs px-2 py-1 rounded">
              {tag}
            </span>
          ))}
        </div>
      </div>
    </Link>
  );
}
