'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

type PaymentFrequency = 'monthly' | 'bi-weekly' | 'accelerated-bi-weekly' | 'weekly' | 'accelerated-weekly';
type AmortizationPeriod = 5 | 10 | 15 | 20 | 25 | 30;

interface AmortizationEntry {
  paymentNumber: number;
  paymentDate: string;
  payment: number;
  principal: number;
  interest: number;
  remainingBalance: number;
  totalInterest: number;
  equityPercentage: number;
}

export default function MortgageCalculator() {
  // Form inputs
  const [homePrice, setHomePrice] = useState<string>('300000');
  const [downPayment, setDownPayment] = useState<string>('60000');
  const [downPaymentPercentage, setDownPaymentPercentage] = useState<number>(20);
  const [loanAmount, setLoanAmount] = useState<string>('240000');
  const [interestRate, setInterestRate] = useState<string>('3.5');
  const [amortizationPeriod, setAmortizationPeriod] = useState<AmortizationPeriod>(25);
  const [paymentFrequency, setPaymentFrequency] = useState<PaymentFrequency>('monthly');
  const [startDate, setStartDate] = useState<string>(new Date().toISOString().split('T')[0]);
  
  // Additional costs
  const [propertyTax, setPropertyTax] = useState<string>('2500');
  const [homeInsurance, setHomeInsurance] = useState<string>('1000');
  const [pmi, setPmi] = useState<string>('0');
  const [hoaFees, setHoaFees] = useState<string>('0');
  
  // Results
  const [monthlyPayment, setMonthlyPayment] = useState<number>(0);
  const [totalInterest, setTotalInterest] = useState<number>(0);
  const [totalPayment, setTotalPayment] = useState<number>(0);
  const [totalCost, setTotalCost] = useState<number>(0);
  const [amortizationSchedule, setAmortizationSchedule] = useState<AmortizationEntry[]>([]);
  const [showAmortization, setShowAmortization] = useState<boolean>(false);
  const [showAdvanced, setShowAdvanced] = useState<boolean>(false);
  
  // Calculate loan details when inputs change
  useEffect(() => {
    calculateMortgage();
  }, [
    homePrice, 
    downPayment, 
    loanAmount, 
    interestRate, 
    amortizationPeriod, 
    paymentFrequency,
    propertyTax,
    homeInsurance,
    pmi,
    hoaFees
  ]);
  
  // Sync down payment percentage and amount
  useEffect(() => {
    const price = parseFloat(homePrice) || 0;
    const down = parseFloat(downPayment) || 0;
    
    if (price > 0) {
      const percentage = (down / price) * 100;
      setDownPaymentPercentage(parseFloat(percentage.toFixed(2)));
      
      // Update loan amount
      const newLoanAmount = Math.max(0, price - down);
      setLoanAmount(newLoanAmount.toFixed(2));
    }
  }, [homePrice, downPayment]);
  
  // Update down payment when percentage changes
  const handleDownPaymentPercentageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const percentage = parseFloat(e.target.value);
    if (!isNaN(percentage) && percentage >= 0 && percentage <= 100) {
      const price = parseFloat(homePrice) || 0;
      const down = (price * percentage) / 100;
      setDownPayment(down.toFixed(2));
      setDownPaymentPercentage(percentage);
    }
  };
  
  const calculateMortgage = () => {
    const principal = parseFloat(loanAmount);
    const rate = parseFloat(interestRate) / 100;
    const years = amortizationPeriod;
    
    if (isNaN(principal) || isNaN(rate) || isNaN(years) || principal <= 0 || years <= 0) {
      resetResults();
      return;
    }
    
    // Determine payments per year and payment multiplier based on frequency
    let paymentsPerYear: number;
    let paymentMultiplier: number = 1;
    
    switch (paymentFrequency) {
      case 'monthly':
        paymentsPerYear = 12;
        break;
      case 'bi-weekly':
        paymentsPerYear = 26;
        break;
      case 'accelerated-bi-weekly':
        paymentsPerYear = 26;
        paymentMultiplier = 0.5; // Half of monthly payment every two weeks
        break;
      case 'weekly':
        paymentsPerYear = 52;
        break;
      case 'accelerated-weekly':
        paymentsPerYear = 52;
        paymentMultiplier = 0.25; // Quarter of monthly payment every week
        break;
      default:
        paymentsPerYear = 12;
    }
    
    const totalPayments = years * paymentsPerYear;
    const periodicRate = rate / paymentsPerYear;
    
    // Calculate payment using the mortgage payment formula
    let payment: number;
    if (periodicRate === 0) {
      payment = principal / totalPayments;
    } else {
      payment = (principal * periodicRate * Math.pow(1 + periodicRate, totalPayments)) / 
                (Math.pow(1 + periodicRate, totalPayments) - 1);
    }
    
    // Adjust for accelerated payment frequencies
    if (paymentFrequency === 'accelerated-bi-weekly') {
      // Calculate monthly payment first, then take half for accelerated bi-weekly
      const monthlyRate = rate / 12;
      const monthlyPayments = years * 12;
      const monthlyPayment = (principal * monthlyRate * Math.pow(1 + monthlyRate, monthlyPayments)) / 
                           (Math.pow(1 + monthlyRate, monthlyPayments) - 1);
      payment = monthlyPayment * 0.5; // Half of monthly payment every two weeks
    } else if (paymentFrequency === 'accelerated-weekly') {
      // Calculate monthly payment first, then take a quarter for accelerated weekly
      const monthlyRate = rate / 12;
      const monthlyPayments = years * 12;
      const monthlyPayment = (principal * monthlyRate * Math.pow(1 + monthlyRate, monthlyPayments)) / 
                           (Math.pow(1 + monthlyRate, monthlyPayments) - 1);
      payment = monthlyPayment * 0.25; // Quarter of monthly payment every week
    }
    
    // Generate amortization schedule
    const schedule: AmortizationEntry[] = [];
    let remainingBalance = principal;
    let totalInterestPaid = 0;
    const startDateObj = new Date(startDate);
    
    for (let i = 1; i <= totalPayments && remainingBalance > 0.01; i++) {
      // Calculate payment date
      const paymentDate = new Date(startDateObj);
      
      switch (paymentFrequency) {
        case 'monthly':
          paymentDate.setMonth(paymentDate.getMonth() + (i - 1));
          break;
        case 'bi-weekly':
        case 'accelerated-bi-weekly':
          paymentDate.setDate(paymentDate.getDate() + ((i - 1) * 14));
          break;
        case 'weekly':
        case 'accelerated-weekly':
          paymentDate.setDate(paymentDate.getDate() + ((i - 1) * 7));
          break;
      }
      
      const interestPayment = remainingBalance * periodicRate;
      let principalPayment = payment - interestPayment;
      
      // For the last payment, adjust to ensure the loan is fully paid
      if (i === totalPayments || principalPayment > remainingBalance) {
        principalPayment = remainingBalance;
      }
      
      remainingBalance -= principalPayment;
      totalInterestPaid += interestPayment;
      
      // Ensure remaining balance doesn't go below 0 due to floating point errors
      if (remainingBalance < 0.01) remainingBalance = 0;
      
      const totalPaid = (parseFloat(homePrice) - parseFloat(loanAmount)) + (parseFloat(loanAmount) - remainingBalance);
      const equityPercentage = (totalPaid / parseFloat(homePrice)) * 100;
      
      schedule.push({
        paymentNumber: i,
        paymentDate: paymentDate.toLocaleDateString(),
        payment: payment,
        principal: principalPayment,
        interest: interestPayment,
        remainingBalance: remainingBalance >= 0 ? remainingBalance : 0,
        totalInterest: totalInterestPaid,
        equityPercentage: equityPercentage > 100 ? 100 : equityPercentage
      });
    }
    
    // Calculate additional monthly costs
    const monthlyPropertyTax = (parseFloat(propertyTax) || 0) / 12;
    const monthlyHomeInsurance = (parseFloat(homeInsurance) || 0) / 12;
    const monthlyPmi = (parseFloat(pmi) || 0) / 12;
    const monthlyHoaFees = parseFloat(hoaFees) || 0;
    
    // Calculate total monthly payment including all costs
    let totalMonthlyPayment = 0;
    switch (paymentFrequency) {
      case 'monthly':
        totalMonthlyPayment = payment + monthlyPropertyTax + monthlyHomeInsurance + monthlyPmi + monthlyHoaFees;
        break;
      case 'bi-weekly':
        totalMonthlyPayment = (payment * 26 / 12) + monthlyPropertyTax + monthlyHomeInsurance + monthlyPmi + monthlyHoaFees;
        break;
      case 'accelerated-bi-weekly':
        // For display purposes, show the equivalent monthly payment (26 payments / 12 months)
        totalMonthlyPayment = (payment * 26 / 12) + monthlyPropertyTax + monthlyHomeInsurance + monthlyPmi + monthlyHoaFees;
        break;
      case 'weekly':
        totalMonthlyPayment = (payment * 52 / 12) + monthlyPropertyTax + monthlyHomeInsurance + monthlyPmi + monthlyHoaFees;
        break;
      case 'accelerated-weekly':
        // For display purposes, show the equivalent monthly payment (52 payments / 12 months)
        totalMonthlyPayment = (payment * 52 / 12) + monthlyPropertyTax + monthlyHomeInsurance + monthlyPmi + monthlyHoaFees;
        break;
    }
    
    // Update state
    setMonthlyPayment(payment);
    setTotalInterest(totalInterestPaid);
    setTotalPayment(principal + totalInterestPaid);
    setTotalCost(parseFloat(homePrice) + totalInterestPaid + (parseFloat(propertyTax) || 0) * amortizationPeriod + 
                (parseFloat(homeInsurance) || 0) * amortizationPeriod + (parseFloat(pmi) || 0) * Math.min(amortizationPeriod, 10));
    setAmortizationSchedule(schedule);
  };
  
  const resetResults = () => {
    setMonthlyPayment(0);
    setTotalInterest(0);
    setTotalPayment(0);
    setTotalCost(0);
    setAmortizationSchedule([]);
  };
  
  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    }).format(amount);
  };
  
  const formatCurrencyWithCents = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2
    }).format(amount);
  };
  
  const toggleAmortization = () => {
    setShowAmortization(!showAmortization);
  };
  
  const toggleAdvanced = () => {
    setShowAdvanced(!showAdvanced);
  };

  const exportToCSV = () => {
    if (amortizationSchedule.length === 0) return;

    // Create CSV header
    const headers = ['Payment #', 'Date', 'Payment', 'Principal', 'Interest', 'Total Interest', 'Remaining Balance', 'Equity %'];
    
    // Create CSV rows
    const rows = amortizationSchedule.map(entry => [
      entry.paymentNumber,
      entry.paymentDate,
      entry.payment.toFixed(2),
      entry.principal.toFixed(2),
      entry.interest.toFixed(2),
      entry.totalInterest.toFixed(2),
      entry.remainingBalance.toFixed(2),
      entry.equityPercentage.toFixed(2)
    ]);

    // Combine headers and rows
    const csvContent = [
      headers.join(','),
      ...rows.map(row => row.join(','))
    ].join('\n');

    // Create blob and download
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    
    link.setAttribute('href', url);
    link.setAttribute('download', `home-loan-amortization-schedule.csv`);
    link.style.visibility = 'hidden';
    
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  const printSchedule = () => {
    if (amortizationSchedule.length === 0) return;

    const printWindow = window.open('', '_blank');
    if (!printWindow) return;

    const htmlContent = `
      <!DOCTYPE html>
      <html>
        <head>
          <title>Home Loan Amortization Schedule</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              padding: 20px;
              color: #333;
            }
            h1 {
              color: #2563eb;
              margin-bottom: 10px;
            }
            .summary {
              background: #f3f4f6;
              padding: 15px;
              border-radius: 8px;
              margin-bottom: 20px;
            }
            .summary-item {
              display: flex;
              justify-content: space-between;
              margin: 5px 0;
            }
            table {
              width: 100%;
              border-collapse: collapse;
              margin-top: 20px;
            }
            th {
              background: #2563eb;
              color: white;
              padding: 10px;
              text-align: left;
              font-size: 12px;
            }
            td {
              padding: 8px;
              border-bottom: 1px solid #e5e7eb;
              font-size: 11px;
            }
            tr:nth-child(even) {
              background: #f9fafb;
            }
            .text-right {
              text-align: right;
            }
            @media print {
              .no-print {
                display: none;
              }
            }
          </style>
        </head>
        <body>
          <h1>Home Loan Amortization Schedule</h1>
          
          <div class="summary">
            <div class="summary-item">
              <strong>Home Price:</strong>
              <span>${formatCurrency(parseFloat(homePrice))}</span>
            </div>
            <div class="summary-item">
              <strong>Deposit:</strong>
              <span>${formatCurrency(parseFloat(downPayment))}</span>
            </div>
            <div class="summary-item">
              <strong>Loan Amount:</strong>
              <span>${formatCurrency(parseFloat(loanAmount))}</span>
            </div>
            <div class="summary-item">
              <strong>Interest Rate:</strong>
              <span>${interestRate}%</span>
            </div>
            <div class="summary-item">
              <strong>Loan Term:</strong>
              <span>${amortizationPeriod} years</span>
            </div>
            <div class="summary-item">
              <strong>Payment Frequency:</strong>
              <span>${paymentFrequency}</span>
            </div>
            <div class="summary-item">
              <strong>Monthly Payment:</strong>
              <span>${formatCurrency(monthlyPayment)}</span>
            </div>
            <div class="summary-item">
              <strong>Total Interest:</strong>
              <span>${formatCurrency(totalInterest)}</span>
            </div>
          </div>

          <table>
            <thead>
              <tr>
                <th>Payment #</th>
                <th>Date</th>
                <th class="text-right">Payment</th>
                <th class="text-right">Principal</th>
                <th class="text-right">Interest</th>
                <th class="text-right">Total Interest</th>
                <th class="text-right">Balance</th>
                <th class="text-right">Equity %</th>
              </tr>
            </thead>
            <tbody>
              ${amortizationSchedule.map(entry => `
                <tr>
                  <td>${entry.paymentNumber}</td>
                  <td>${entry.paymentDate}</td>
                  <td class="text-right">${formatCurrencyWithCents(entry.payment)}</td>
                  <td class="text-right">${formatCurrencyWithCents(entry.principal)}</td>
                  <td class="text-right">${formatCurrencyWithCents(entry.interest)}</td>
                  <td class="text-right">${formatCurrencyWithCents(entry.totalInterest)}</td>
                  <td class="text-right">${formatCurrency(entry.remainingBalance)}</td>
                  <td class="text-right">${entry.equityPercentage.toFixed(1)}%</td>
                </tr>
              `).join('')}
            </tbody>
          </table>

          <button class="no-print" onclick="window.print()" style="margin-top: 20px; padding: 10px 20px; background: #2563eb; color: white; border: none; border-radius: 5px; cursor: pointer;">
            Print Schedule
          </button>
        </body>
      </html>
    `;

    printWindow.document.write(htmlContent);
    printWindow.document.close();
  };
  
  // Calculate loan-to-value ratio
  const ltvRatio = parseFloat(homePrice) > 0 
    ? ((parseFloat(loanAmount) / parseFloat(homePrice)) * 100).toFixed(2) + '%' 
    : '0%';
  
  // Calculate down payment percentage
  const calculatedDownPaymentPercentage = parseFloat(homePrice) > 0 
    ? ((parseFloat(downPayment) / parseFloat(homePrice)) * 100).toFixed(2) + '%' 
    : '0%';
  
  // Calculate property tax rate
  const propertyTaxRate = parseFloat(homePrice) > 0 
    ? ((parseFloat(propertyTax) / parseFloat(homePrice)) * 100).toFixed(2) + '%' 
    : '0%';

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Home Loan Calculator</h1>
        <p className="mt-2 text-gray-600">
          Estimate your monthly home loan repayments, including rates, levies, and insurance costs.
        </p>
      </div>

      {/* Top Banner Ad */}
      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label htmlFor="home-price" className="block text-sm font-medium text-gray-700 mb-1">
                  Home Price
                </label>
                <div className="relative rounded-md shadow-sm">
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                    <span className="text-gray-500 sm:text-sm">R</span>
                  </div>
                  <input
                    type="text"
                    id="home-price"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-3 sm:text-sm border-gray-300 rounded-md border"
                    placeholder="300,000"
                    value={homePrice}
                    onChange={(e) => setHomePrice(e.target.value.replace(/[^0-9]/g, ''))}
                    inputMode="numeric"
                  />
                </div>
              </div>
              
              <div>
                <div className="flex justify-between">
                  <label htmlFor="down-payment" className="block text-sm font-medium text-gray-700 mb-1">
                    Down Payment
                  </label>
                  <span className="text-xs text-gray-500">{calculatedDownPaymentPercentage}</span>
                </div>
                <div className="relative rounded-md shadow-sm">
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                    <span className="text-gray-500 sm:text-sm">R</span>
                  </div>
                  <input
                    type="text"
                    id="down-payment"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-3 sm:text-sm border-gray-300 rounded-md border"
                    placeholder="60,000"
                    value={downPayment}
                    onChange={(e) => setDownPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    inputMode="decimal"
                  />
                </div>
                <div className="mt-2">
                  <input
                    type="range"
                    min="0"
                    max="100"
                    step="0.5"
                    value={downPaymentPercentage}
                    onChange={handleDownPaymentPercentageChange}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                  />
                </div>
              </div>
              
              <div>
                <label htmlFor="loan-amount" className="block text-sm font-medium text-gray-700 mb-1">
                  Loan Amount
                </label>
                <div className="relative rounded-md shadow-sm">
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                    <span className="text-gray-500 sm:text-sm">R</span>
                  </div>
                  <input
                    type="text"
                    id="loan-amount"
                    className="bg-gray-100 block w-full pl-7 pr-12 py-3 sm:text-sm border-gray-300 rounded-md border"
                    value={formatCurrency(parseFloat(loanAmount) || 0).replace('$', '').replace(/,/g, '')}
                    readOnly
                  />
                </div>
                <div className="text-xs text-gray-500 mt-1">
                  LTV: {ltvRatio} (Loan-to-Value)
                </div>
              </div>
              
              <div>
                <label htmlFor="interest-rate" className="block text-sm font-medium text-gray-700 mb-1">
                  Interest Rate
                </label>
                <div className="relative rounded-md shadow-sm">
                  <input
                    type="text"
                    id="interest-rate"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full pr-12 py-3 sm:text-sm border-gray-300 rounded-md border pl-3"
                    placeholder="3.5"
                    value={interestRate}
                    onChange={(e) => setInterestRate(e.target.value.replace(/[^0-9.]/g, ''))}
                    inputMode="decimal"
                  />
                  <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                    <span className="text-gray-500 sm:text-sm">%</span>
                  </div>
                </div>
              </div>
              
              <div>
                <label htmlFor="amortization" className="block text-sm font-medium text-gray-700 mb-1">
                  Amortization Period
                </label>
                <select
                  id="amortization"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full py-3 sm:text-sm border-gray-300 rounded-md border"
                  value={amortizationPeriod}
                  onChange={(e) => setAmortizationPeriod(parseInt(e.target.value) as AmortizationPeriod)}
                >
                  <option value="5">5 years</option>
                  <option value="10">10 years</option>
                  <option value="15">15 years</option>
                  <option value="20">20 years</option>
                  <option value="25">25 years</option>
                  <option value="30">30 years</option>
                </select>
              </div>
              
              <div>
                <label htmlFor="payment-frequency" className="block text-sm font-medium text-gray-700 mb-1">
                  Payment Frequency
                </label>
                <select
                  id="payment-frequency"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full py-3 sm:text-sm border-gray-300 rounded-md border"
                  value={paymentFrequency}
                  onChange={(e) => setPaymentFrequency(e.target.value as PaymentFrequency)}
                >
                  <option value="monthly">Monthly</option>
                  <option value="bi-weekly">Bi-Weekly</option>
                  <option value="accelerated-bi-weekly">Accelerated Bi-Weekly</option>
                  <option value="weekly">Weekly</option>
                  <option value="accelerated-weekly">Accelerated Weekly</option>
                </select>
              </div>
            </div>
            
            {/* Advanced Options */}
            <div className="mt-6">
              <button
                type="button"
                onClick={toggleAdvanced}
                className="text-sm font-medium text-blue-600 hover:text-blue-500 focus:outline-none"
              >
                {showAdvanced ? 'Hide' : 'Show'} Advanced Options
              </button>
              
              {showAdvanced && (
                <div className="mt-4 p-4 bg-gray-50 rounded-lg space-y-4">
                  <h3 className="text-sm font-medium text-gray-700">Additional Costs</h3>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <label htmlFor="property-tax" className="block text-xs font-medium text-gray-700 mb-1">
                        Annual Municipal Rates
                      </label>
                      <div className="relative rounded-md shadow-sm">
                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">R</span>
                        </div>
                        <input
                          type="text"
                          id="property-tax"
                          className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-2 text-xs border-gray-300 rounded-md border"
                          placeholder="2,500"
                          value={propertyTax}
                          onChange={(e) => setPropertyTax(e.target.value.replace(/[^0-9.]/g, ''))}
                          inputMode="decimal"
                        />
                      </div>
                      <div className="text-xs text-gray-500 mt-1">
                        {propertyTaxRate} of home price
                      </div>
                    </div>
                    
                    <div>
                      <label htmlFor="home-insurance" className="block text-xs font-medium text-gray-700 mb-1">
                        Annual Home Insurance
                      </label>
                      <div className="relative rounded-md shadow-sm">
                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">R</span>
                        </div>
                        <input
                          type="text"
                          id="home-insurance"
                          className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-2 text-xs border-gray-300 rounded-md border"
                          placeholder="1,000"
                          value={homeInsurance}
                          onChange={(e) => setHomeInsurance(e.target.value.replace(/[^0-9.]/g, ''))}
                          inputMode="decimal"
                        />
                      </div>
                    </div>
                    
                    <div>
                      <label htmlFor="pmi" className="block text-xs font-medium text-gray-700 mb-1">
                        Bond Insurance (if &lt;20% deposit)
                      </label>
                      <div className="relative rounded-md shadow-sm">
                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">R</span>
                        </div>
                        <input
                          type="text"
                          id="pmi"
                          className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-2 text-xs border-gray-300 rounded-md border"
                          placeholder="0"
                          value={pmi}
                          onChange={(e) => setPmi(e.target.value.replace(/[^0-9.]/g, ''))}
                          inputMode="decimal"
                        />
                        <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">/year</span>
                        </div>
                      </div>
                    </div>
                    
                    <div>
                      <label htmlFor="hoa-fees" className="block text-xs font-medium text-gray-700 mb-1">
                        Monthly Levies
                      </label>
                      <div className="relative rounded-md shadow-sm">
                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">R</span>
                        </div>
                        <input
                          type="text"
                          id="hoa-fees"
                          className="focus:ring-blue-500 focus:border-blue-500 block w-full pl-7 pr-12 py-2 text-xs border-gray-300 rounded-md border"
                          placeholder="0"
                          value={hoaFees}
                          onChange={(e) => setHoaFees(e.target.value.replace(/[^0-9.]/g, ''))}
                          inputMode="decimal"
                        />
                        <div className="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                          <span className="text-gray-500 text-xs">/month</span>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  <div className="pt-2">
                    <label htmlFor="start-date" className="block text-xs font-medium text-gray-700 mb-1">
                      Start Date
                    </label>
                    <input
                      type="date"
                      id="start-date"
                      className="focus:ring-blue-500 focus:border-blue-500 block w-full py-2 px-3 text-xs border border-gray-300 rounded-md"
                      value={startDate}
                      onChange={(e) => setStartDate(e.target.value)}
                    />
                  </div>
                </div>
              )}
            </div>
          </div>
          
          {/* Right Column - Results */}
          <div className="lg:col-span-1">
            <div className="bg-blue-50 rounded-lg p-6 h-full">
              <h2 className="text-lg font-semibold text-gray-900 mb-4">Mortgage Summary</h2>
              
              {monthlyPayment > 0 ? (
                <div className="space-y-4">
                  <div className="bg-white p-4 rounded-lg shadow-sm">
                    <div className="text-sm font-medium text-gray-500">Estimated Payment</div>
                    <div className="text-2xl font-bold text-blue-600 mt-1">
                      {formatCurrencyWithCents(monthlyPayment)}
                      <span className="text-sm font-normal text-gray-500"> / month</span>
                    </div>
                    <div className="text-xs text-gray-500 mt-1">
                      {paymentFrequency === 'monthly' ? '12' : 
                       paymentFrequency === 'bi-weekly' || paymentFrequency === 'accelerated-bi-weekly' ? '26' : '52'} payments per year
                    </div>
                  </div>
                  
                  <div className="space-y-3 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Principal & Interest:</span>
                      <span className="font-medium">{formatCurrencyWithCents(monthlyPayment)}</span>
                    </div>
                    
                    {(parseFloat(propertyTax) > 0 || parseFloat(homeInsurance) > 0 || parseFloat(pmi) > 0 || parseFloat(hoaFees) > 0) && (
                      <div className="pt-2 border-t border-gray-200">
                        <div className="text-xs font-medium text-gray-500 mb-1">Additional Monthly Costs</div>
                        
                        {parseFloat(propertyTax) > 0 && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">Property Tax:</span>
                            <span className="font-medium">{formatCurrencyWithCents(parseFloat(propertyTax) / 12)}</span>
                          </div>
                        )}
                        
                        {parseFloat(homeInsurance) > 0 && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">Home Insurance:</span>
                            <span className="font-medium">{formatCurrencyWithCents(parseFloat(homeInsurance) / 12)}</span>
                          </div>
                        )}
                        
                        {parseFloat(pmi) > 0 && parseFloat(downPayment) / parseFloat(homePrice) < 0.2 && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">PMI:</span>
                            <span className="font-medium">{formatCurrencyWithCents(parseFloat(pmi) / 12)}</span>
                          </div>
                        )}
                        
                        {parseFloat(hoaFees) > 0 && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">HOA Fees:</span>
                            <span className="font-medium">{formatCurrencyWithCents(parseFloat(hoaFees))}</span>
                          </div>
                        )}
                        
                        <div className="flex justify-between font-medium pt-1 border-t border-gray-200 mt-2">
                          <span>Total Monthly Payment:</span>
                          <span>
                            {formatCurrencyWithCents(
                              monthlyPayment + 
                              (parseFloat(propertyTax) || 0) / 12 + 
                              (parseFloat(homeInsurance) || 0) / 12 + 
                              (parseFloat(pmi) > 0 && parseFloat(downPayment) / parseFloat(homePrice) < 0.2 ? parseFloat(pmi) / 12 : 0) + 
                              (parseFloat(hoaFees) || 0)
                            )}
                          </span>
                        </div>
                      </div>
                    )}
                  </div>
                  
                  <div className="pt-4 mt-4 border-t border-gray-200">
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Total Interest Paid:</span>
                      <span className="font-medium">{formatCurrency(totalInterest)}</span>
                    </div>
                    <div className="flex justify-between text-sm mt-1">
                      <span className="text-gray-600">Total of {amortizationSchedule.length} Payments:</span>
                      <span className="font-medium">{formatCurrency(totalPayment)}</span>
                    </div>
                    <div className="flex justify-between text-sm mt-1">
                      <span className="text-gray-600">Total Cost (Home + Interest + Fees):</span>
                      <span className="font-medium">{formatCurrency(totalCost)}</span>
                    </div>
                  </div>
                  
                  <div className="pt-4">
                    <div className="h-2 w-full bg-gray-200 rounded-full overflow-hidden">
                      <div 
                        className="h-full bg-blue-600 rounded-full" 
                        style={{ 
                          width: `${(parseFloat(loanAmount) / totalPayment) * 100}%`,
                          maxWidth: '100%'
                        }}
                      ></div>
                    </div>
                    <div className="flex justify-between mt-2 text-xs text-gray-500">
                      <span>Principal: {formatCurrency(parseFloat(loanAmount))}</span>
                      <span>Interest: {formatCurrency(totalInterest)}</span>
                    </div>
                  </div>
                  
                  <button
                    type="button"
                    onClick={toggleAmortization}
                    className="mt-4 w-full bg-white py-2 px-4 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                  >
                    {showAmortization ? 'Hide' : 'View'} Amortization Schedule
                  </button>
                  
                  <div className="text-xs text-gray-500 mt-2">
                    <p>Interest rate: {interestRate}%</p>
                    <p>Term: {amortizationPeriod} years</p>
                    <p>Payment frequency: {paymentFrequency.replace(/-/g, ' ')}</p>
                  </div>
                </div>
              ) : (
                <div className="text-center py-8">
                  <svg
                    className="mx-auto h-12 w-12 text-gray-400"
                    fill="none"
                    viewBox="0 0 24 24"
                    stroke="currentColor"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={1}
                      d="M3 10h18v7a2 2 0 01-2 2H5a2 2 0 01-2-2v-7z"
                    />
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={1}
                      d="M12 10v8m-8-8a8 8 0 0116 0"
                    />
                  </svg>
                  <h3 className="mt-2 text-sm font-medium text-gray-900">Enter loan details</h3>
                  <p className="mt-1 text-sm text-gray-500">Fill in the form to see your payment details.</p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
      
      {/* Amortization Schedule */}
      {showAmortization && amortizationSchedule.length > 0 && (
        <div className="bg-white rounded-xl shadow-md overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Amortization Schedule</h3>
            <p className="mt-1 text-sm text-gray-500">
              A complete table of periodic mortgage payments, showing the amount of principal and interest that comprises each payment.
            </p>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Payment #
                  </th>
                  <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Date
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Payment
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Principal
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Interest
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Total Interest
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Balance
                  </th>
                  <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Equity
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {amortizationSchedule.slice(0, 12).map((entry, index) => (
                  <tr key={entry.paymentNumber} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-gray-500">
                      {entry.paymentNumber}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-gray-500">
                      {entry.paymentDate}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900 font-medium">
                      {formatCurrencyWithCents(entry.payment)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900">
                      {formatCurrencyWithCents(entry.principal)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-500">
                      {formatCurrencyWithCents(entry.interest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-500">
                      {formatCurrencyWithCents(entry.totalInterest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900 font-medium">
                      {formatCurrency(entry.remainingBalance)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900">
                      {entry.equityPercentage.toFixed(1)}%
                    </td>
                  </tr>
                ))}
                
                {/* Show a summary row if there are more than 12 payments */}
                {amortizationSchedule.length > 12 && (
                  <tr className="bg-gray-50">
                    <td colSpan={8} className="px-6 py-2 text-center text-xs text-gray-500">
                      ... {amortizationSchedule.length - 12} more payments ...
                    </td>
                  </tr>
                )}
                
                {/* Show the last 12 payments if there are more than 24 total */}
                {amortizationSchedule.length > 24 && amortizationSchedule.slice(-12).map((entry, index) => (
                  <tr key={`last-${entry.paymentNumber}`} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-gray-500">
                      {entry.paymentNumber}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-gray-500">
                      {entry.paymentDate}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900 font-medium">
                      {formatCurrencyWithCents(entry.payment)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900">
                      {formatCurrencyWithCents(entry.principal)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-500">
                      {formatCurrencyWithCents(entry.interest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-500">
                      {formatCurrencyWithCents(entry.totalInterest)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900 font-medium">
                      {formatCurrency(entry.remainingBalance)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-xs text-right text-gray-900">
                      {entry.equityPercentage.toFixed(1)}%
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          <div className="bg-gray-50 px-6 py-3 flex justify-between items-center border-t border-gray-200">
            <div className="text-sm text-gray-700">
              Showing {Math.min(12, amortizationSchedule.length)} of {amortizationSchedule.length} payments
            </div>
            <div className="flex space-x-2">
              <button
                type="button"
                onClick={exportToCSV}
                className="inline-flex items-center px-3 py-1.5 border border-gray-300 shadow-sm text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
              >
                <svg className="-ml-0.5 mr-1.5 h-4 w-4" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
                </svg>
                Export to CSV
              </button>
              <button
                type="button"
                onClick={printSchedule}
                className="inline-flex items-center px-3 py-1.5 border border-transparent text-xs font-medium rounded shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
              >
                <svg className="-ml-0.5 mr-1.5 h-4 w-4" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 17h2a2 2 0 002-2v-4a2 2 0 00-2-2H5a2 2 0 00-2 2v4a2 2 0 002 2h2m2 4h6a2 2 0 002-2v-4a2 2 0 00-2-2H9a2 2 0 00-2 2v4a2 2 0 002 2zm8-12V5a2 2 0 00-2-2H9a2 2 0 00-2 2v4h10z" />
                </svg>
                Print Schedule
              </button>
            </div>
          </div>
        </div>
      )}
      
      {/* In-Content Ad */}
      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>How to Use the Mortgage Calculator</h2>
        <p>
          Our mortgage calculator helps you estimate your monthly mortgage payments, including principal, interest, property taxes, 
          home insurance, and other costs. Here's how to use it:
        </p>
        <ol>
          <li><strong>Enter the home price</strong> - The total purchase price of the home you're considering.</li>
          <li><strong>Input your down payment</strong> - Either as a dollar amount or use the slider to set a percentage.</li>
          <li><strong>Adjust the loan term</strong> - Typically 15 or 30 years, but other terms are available.</li>
          <li><strong>Enter the interest rate</strong> - Current market rates or your quoted rate from a lender.</li>
          <li><strong>Select payment frequency</strong> - Monthly, bi-weekly, or weekly payments.</li>
          <li><strong>Add additional costs</strong> - Click "Show Advanced Options" to include property taxes, insurance, PMI, and HOA fees.</li>
        </ol>

        <h2>Understanding Your Mortgage Payment</h2>
        <p>
          Your monthly mortgage payment typically consists of four components, often referred to as PITI:
        </p>
        <ul>
          <li><strong>Principal</strong> - The amount you borrow to buy the home, which you pay back over time.</li>
          <li><strong>Interest</strong> - The cost of borrowing the money, calculated as a percentage of your loan amount.</li>
          <li><strong>Taxes</strong> - Property taxes assessed by your local government, usually paid through an escrow account.</li>
          <li><strong>Insurance</strong> - Typically includes homeowners insurance and may include private mortgage insurance (PMI) if your down payment is less than 20%.</li>
        </ul>

        <h2>Mortgage Payment Frequency Options</h2>
        <p>
          You can choose different payment frequencies, which can affect how much interest you pay over the life of your loan:
        </p>
        <ul>
          <li><strong>Monthly</strong> - One payment per month (12 payments per year)</li>
          <li><strong>Bi-Weekly</strong> - Half of your monthly payment every two weeks (26 half-payments = 13 monthly payments per year)</li>
          <li><strong>Accelerated Bi-Weekly</strong> - Half of what your monthly payment would be, paid every two weeks (26 half-payments = 13 monthly payments per year)</li>
          <li><strong>Weekly</strong> - One quarter of your monthly payment every week (52 quarter-payments = 13 monthly payments per year)</li>
          <li><strong>Accelerated Weekly</strong> - One quarter of what your monthly payment would be, paid every week (52 quarter-payments = 13 monthly payments per year)</li>
        </ul>

        <h2>Amortization Schedule</h2>
        <p>
          The amortization schedule shows how each payment is split between principal and interest over the life of your loan. 
          In the early years, most of your payment goes toward interest. As you pay down the principal, more of each payment 
          goes toward paying down the loan balance.
        </p>

        <h2>Tips for Saving on Your Mortgage</h2>
        <ul>
          <li><strong>Make extra payments</strong> - Even small additional payments can significantly reduce the total interest you pay.</li>
          <li>Consider a shorter loan term - A 15-year mortgage typically has a lower interest rate than a 30-year loan.</li>
          <li>Shop around for the best rates - Even a small difference in interest rates can save you thousands over the life of your loan.</li>
          <li>Consider making bi-weekly payments - This results in one extra monthly payment each year, which can shorten your loan term.</li>
          <li>Pay points to lower your rate - Paying points (upfront fees) can lower your interest rate over the life of the loan.</li>
        </ul>

        <h2>Frequently Asked Questions</h2>
        <div className="space-y-4">
          <div>
            <h3 className="font-semibold">What's included in my monthly mortgage payment?</h3>
            <p className="text-gray-700">
              A typical mortgage payment includes principal, interest, property taxes, and homeowners insurance (PITI). 
              If your down payment was less than 20%, you may also pay private mortgage insurance (PMI). 
              If you have an HOA, those fees would be in addition to your mortgage payment.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">What's the difference between a fixed-rate and adjustable-rate mortgage (ARM)?</h3>
            <p className="text-gray-700">
              A fixed-rate mortgage has the same interest rate for the entire loan term, so your monthly payment stays the same. 
              An ARM has an initial fixed-rate period (often 5, 7, or 10 years) after which the rate adjusts periodically 
              (typically annually) based on market conditions. ARMs often start with lower rates but can increase over time.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">How much should I put down on a house?</h3>
            <p className="text-gray-700">
              The traditional recommendation is 20% of the home's purchase price, which allows you to avoid PMI. 
              However, many loans are available with as little as 3-5% down. Consider your savings, monthly budget, 
              and long-term financial goals when deciding how much to put down.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">How does my credit score affect my mortgage rate?</h3>
            <p className="text-gray-700">
              Generally, the higher your credit score, the lower your interest rate will be. Lenders view borrowers with 
              higher credit scores as less risky. Even a small difference in interest rates can save you thousands of 
              dollars over the life of your loan, so it's worth improving your credit score before applying for a mortgage.
            </p>
          </div>
          
          <div>
            <h3 className="font-semibold">What is PMI and how can I avoid it?</h3>
            <p className="text-gray-700">
              Private Mortgage Insurance (PMI) is typically required when you make a down payment of less than 20% of the 
              home's purchase price. PMI protects the lender if you default on your loan. You can avoid PMI by making a 
              20% down payment, using a piggyback loan, or opting for a lender-paid mortgage insurance (LPMI) option.
            </p>
          </div>
        </div>
      </div>
      
      {/* Bottom CTA */}
      <div className="bg-blue-50 rounded-lg p-6 text-center">
        <h2 className="text-xl font-bold mb-2">Ready to Take the Next Step?</h2>
        <p className="text-gray-700 mb-4 max-w-2xl mx-auto">
          Get pre-approved for a mortgage and find out how much home you can afford. Our mortgage specialists are here to help you 
          find the perfect loan for your situation.
        </p>
        <div className="flex flex-col sm:flex-row justify-center gap-4">
          <button className="bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-6 rounded-lg transition duration-300 inline-flex items-center justify-center">
            <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 10h18v7a2 2 0 01-2 2H5a2 2 0 01-2-2v-7z" />
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v8m-8-8a8 8 0 0116 0" />
            </svg>
            Get Pre-Approved
          </button>
          <button className="bg-white hover:bg-gray-50 text-blue-600 font-semibold py-2 px-6 border border-blue-300 rounded-lg transition duration-300 inline-flex items-center justify-center">
            <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z" />
            </svg>
            Speak with a Loan Officer
          </button>
        </div>
      </div>
    </div>
  );
}
