'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export default function LoanToIncomeCalculator() {
  const [monthlyIncome, setMonthlyIncome] = useState<string>('25000');
  const [homeLoanPayment, setHomeLoanPayment] = useState<string>('8000');
  const [carLoanPayment, setCarLoanPayment] = useState<string>('3500');
  const [creditCardPayment, setCreditCardPayment] = useState<string>('1200');
  const [personalLoanPayment, setPersonalLoanPayment] = useState<string>('1500');
  const [otherDebtPayment, setOtherDebtPayment] = useState<string>('500');
  
  const [totalDebt, setTotalDebt] = useState<number>(0);
  const [dtiRatio, setDtiRatio] = useState<number>(0);
  const [maxAffordableDebt, setMaxAffordableDebt] = useState<number>(0);
  const [remainingCapacity, setRemainingCapacity] = useState<number>(0);
  const [riskLevel, setRiskLevel] = useState<string>('');

  useEffect(() => {
    calculateDTI();
  }, [monthlyIncome, homeLoanPayment, carLoanPayment, creditCardPayment, personalLoanPayment, otherDebtPayment]);

  const calculateDTI = () => {
    const income = parseFloat(monthlyIncome) || 0;
    const homeLoan = parseFloat(homeLoanPayment) || 0;
    const carLoan = parseFloat(carLoanPayment) || 0;
    const creditCard = parseFloat(creditCardPayment) || 0;
    const personalLoan = parseFloat(personalLoanPayment) || 0;
    const otherDebt = parseFloat(otherDebtPayment) || 0;

    const totalDebtPayments = homeLoan + carLoan + creditCard + personalLoan + otherDebt;
    
    if (income > 0) {
      const ratio = (totalDebtPayments / income) * 100;
      const maxDebt = income * 0.36; // 36% is generally considered max
      const remaining = maxDebt - totalDebtPayments;
      
      setTotalDebt(totalDebtPayments);
      setDtiRatio(ratio);
      setMaxAffordableDebt(maxDebt);
      setRemainingCapacity(remaining);
      
      // Determine risk level
      if (ratio <= 20) {
        setRiskLevel('Excellent');
      } else if (ratio <= 30) {
        setRiskLevel('Good');
      } else if (ratio <= 36) {
        setRiskLevel('Fair');
      } else if (ratio <= 43) {
        setRiskLevel('High');
      } else {
        setRiskLevel('Very High');
      }
    } else {
      resetResults();
    }
  };

  const resetResults = () => {
    setTotalDebt(0);
    setDtiRatio(0);
    setMaxAffordableDebt(0);
    setRemainingCapacity(0);
    setRiskLevel('');
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getRiskColor = () => {
    if (dtiRatio <= 20) return 'text-green-600';
    if (dtiRatio <= 30) return 'text-blue-600';
    if (dtiRatio <= 36) return 'text-yellow-600';
    if (dtiRatio <= 43) return 'text-orange-600';
    return 'text-red-600';
  };

  const getRiskBgColor = () => {
    if (dtiRatio <= 20) return 'bg-green-100';
    if (dtiRatio <= 30) return 'bg-blue-100';
    if (dtiRatio <= 36) return 'bg-yellow-100';
    if (dtiRatio <= 43) return 'bg-orange-100';
    return 'bg-red-100';
  };

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Debt-to-Income Ratio Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate your debt-to-income ratio to assess your borrowing capacity for South African lenders.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div>
              <label htmlFor="monthly-income" className="block text-sm font-medium text-gray-700 mb-1">
                Gross Monthly Income (R)
              </label>
              <input
                type="text"
                id="monthly-income"
                className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-3 text-sm border-gray-300 rounded-md border"
                value={monthlyIncome}
                onChange={(e) => setMonthlyIncome(e.target.value.replace(/[^0-9.]/g, ''))}
                placeholder="25000"
              />
              <p className="mt-1 text-xs text-gray-500">Before tax and deductions</p>
            </div>

            <div className="border-t pt-4">
              <h3 className="text-md font-semibold text-gray-900 mb-4">Monthly Debt Payments</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label htmlFor="home-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Home Loan / Bond Payment
                  </label>
                  <input
                    type="text"
                    id="home-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={homeLoanPayment}
                    onChange={(e) => setHomeLoanPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="8000"
                  />
                </div>

                <div>
                  <label htmlFor="car-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Car Loan / Vehicle Finance
                  </label>
                  <input
                    type="text"
                    id="car-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={carLoanPayment}
                    onChange={(e) => setCarLoanPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="3500"
                  />
                </div>

                <div>
                  <label htmlFor="credit-card" className="block text-sm font-medium text-gray-700 mb-1">
                    Credit Card Minimum Payment
                  </label>
                  <input
                    type="text"
                    id="credit-card"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={creditCardPayment}
                    onChange={(e) => setCreditCardPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="1200"
                  />
                </div>

                <div>
                  <label htmlFor="personal-loan" className="block text-sm font-medium text-gray-700 mb-1">
                    Personal Loan Payment
                  </label>
                  <input
                    type="text"
                    id="personal-loan"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={personalLoanPayment}
                    onChange={(e) => setPersonalLoanPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="1500"
                  />
                </div>

                <div className="md:col-span-2">
                  <label htmlFor="other-debt" className="block text-sm font-medium text-gray-700 mb-1">
                    Other Debt Payments (Store Cards, etc.)
                  </label>
                  <input
                    type="text"
                    id="other-debt"
                    className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                    value={otherDebtPayment}
                    onChange={(e) => setOtherDebtPayment(e.target.value.replace(/[^0-9.]/g, ''))}
                    placeholder="500"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className={`rounded-lg p-6 ${getRiskBgColor()}`}>
              <h3 className="text-lg font-semibold text-gray-900 mb-4">DTI Analysis</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Debt-to-Income Ratio</p>
                  <p className={`text-3xl font-bold ${getRiskColor()}`}>{dtiRatio.toFixed(1)}%</p>
                </div>

                <div className="border-t border-gray-300 pt-3">
                  <p className="text-xs text-gray-600">Risk Level</p>
                  <p className={`text-lg font-bold ${getRiskColor()}`}>{riskLevel}</p>
                </div>

                <div className="border-t border-gray-300 pt-3">
                  <p className="text-xs text-gray-600">Total Monthly Debt</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(totalDebt)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Monthly Income</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(parseFloat(monthlyIncome) || 0)}</p>
                </div>

                <div className="border-t border-gray-300 pt-3">
                  <p className="text-xs text-gray-600">Max Recommended Debt (36%)</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(maxAffordableDebt)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Remaining Capacity</p>
                  <p className={`text-sm font-semibold ${remainingCapacity >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                    {formatCurrency(Math.abs(remainingCapacity))}
                    {remainingCapacity < 0 && ' over limit'}
                  </p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <h4 className="font-semibold text-gray-900 mb-2">DTI Guidelines</h4>
              <ul className="space-y-1 text-gray-600 text-xs">
                <li>• ≤20%: Excellent</li>
                <li>• 21-30%: Good</li>
                <li>• 31-36%: Fair</li>
                <li>• 37-43%: High Risk</li>
                <li>• &gt;43%: Very High Risk</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Breakdown Chart */}
        <div className="mt-8">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Debt Breakdown</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4">
            {[
              { label: 'Home Loan', value: parseFloat(homeLoanPayment) || 0, color: 'bg-blue-500' },
              { label: 'Car Loan', value: parseFloat(carLoanPayment) || 0, color: 'bg-green-500' },
              { label: 'Credit Card', value: parseFloat(creditCardPayment) || 0, color: 'bg-yellow-500' },
              { label: 'Personal Loan', value: parseFloat(personalLoanPayment) || 0, color: 'bg-purple-500' },
              { label: 'Other Debt', value: parseFloat(otherDebtPayment) || 0, color: 'bg-red-500' },
            ].map((item, index) => (
              <div key={index} className="bg-white border border-gray-200 rounded-lg p-4">
                <div className={`w-3 h-3 ${item.color} rounded-full mb-2`}></div>
                <p className="text-xs text-gray-600">{item.label}</p>
                <p className="text-sm font-semibold text-gray-900">{formatCurrency(item.value)}</p>
                <p className="text-xs text-gray-500">
                  {totalDebt > 0 ? ((item.value / totalDebt) * 100).toFixed(1) : 0}%
                </p>
              </div>
            ))}
          </div>
        </div>
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Debt-to-Income Ratio in South Africa</h2>
        <p>
          Your debt-to-income (DTI) ratio is a key metric that South African lenders use to assess your ability to manage 
          monthly payments and repay borrowed money. It compares your total monthly debt payments to your gross monthly income.
        </p>

        <h2>How to Calculate DTI Ratio</h2>
        <p>
          The formula is simple: <strong>DTI = (Total Monthly Debt Payments ÷ Gross Monthly Income) × 100</strong>
        </p>
        <p>
          For example, if your total monthly debt payments are R14,700 and your gross monthly income is R25,000, 
          your DTI ratio would be 58.8% - which is considered very high risk.
        </p>

        <h2>What South African Lenders Look For</h2>
        <ul>
          <li><strong>Below 20%:</strong> Excellent - You have strong financial health and significant borrowing capacity</li>
          <li><strong>20-30%:</strong> Good - Healthy debt levels, good chance of loan approval</li>
          <li><strong>30-36%:</strong> Fair - Approaching upper limits, may face stricter lending criteria</li>
          <li><strong>36-43%:</strong> High Risk - Difficult to get new credit, may need debt consolidation</li>
          <li><strong>Above 43%:</strong> Very High Risk - Unlikely to qualify for new credit, urgent action needed</li>
        </ul>

        <h2>National Credit Act (NCA) Requirements</h2>
        <p>
          In South Africa, the National Credit Act requires lenders to conduct affordability assessments. 
          Key considerations include:
        </p>
        <ul>
          <li><strong>Affordability Assessment:</strong> Lenders must verify you can afford loan repayments</li>
          <li><strong>Income Verification:</strong> Proof of income (payslips, bank statements) required</li>
          <li><strong>Existing Debt:</strong> All current debt obligations must be considered</li>
          <li><strong>Living Expenses:</strong> Reasonable living costs must be factored in</li>
          <li><strong>Credit Bureau Check:</strong> Your credit score and history will be reviewed</li>
        </ul>

        <h2>What Counts as Debt?</h2>
        <p>
          When calculating your DTI ratio, include all recurring monthly debt payments:
        </p>
        <ul>
          <li>Home loan / bond payments (including rates and levies)</li>
          <li>Vehicle finance / car loan payments</li>
          <li>Credit card minimum payments</li>
          <li>Personal loans</li>
          <li>Store cards and accounts</li>
          <li>Student loans</li>
          <li>Furniture and appliance finance</li>
        </ul>
        <p>
          <strong>Do NOT include:</strong> Utilities, groceries, insurance, medical aid, or other living expenses 
          (though lenders will consider these separately in affordability assessments).
        </p>

        <h2>Improving Your DTI Ratio</h2>
        <p>
          If your DTI ratio is too high, consider these strategies:
        </p>
        <ul>
          <li><strong>Pay Down Debt:</strong> Focus on high-interest debt first (credit cards, store cards)</li>
          <li><strong>Debt Consolidation:</strong> Combine multiple debts into one lower-interest loan</li>
          <li><strong>Increase Income:</strong> Side hustles, overtime, or salary negotiations</li>
          <li><strong>Avoid New Debt:</strong> Don't take on additional credit until DTI improves</li>
          <li><strong>Budget Strictly:</strong> Cut unnecessary expenses and redirect to debt repayment</li>
          <li><strong>Debt Review:</strong> Consider formal debt counseling if severely over-indebted</li>
        </ul>

        <h2>Impact on Loan Applications</h2>
        <p>
          Your DTI ratio directly affects:
        </p>
        <ul>
          <li><strong>Loan Approval:</strong> Higher DTI = lower chance of approval</li>
          <li><strong>Interest Rates:</strong> Better DTI may qualify you for lower rates</li>
          <li><strong>Loan Amount:</strong> Lower DTI allows for larger loan amounts</li>
          <li><strong>Terms:</strong> Better DTI may result in more favorable loan terms</li>
        </ul>

        <h2>Debt Review and Debt Counseling</h2>
        <p>
          If you're struggling with debt (DTI above 50%), consider:
        </p>
        <ul>
          <li><strong>Debt Counseling:</strong> Free advice from registered debt counselors</li>
          <li><strong>Debt Review:</strong> Legal process under NCA to restructure debt</li>
          <li><strong>Payment Plans:</strong> Negotiate with creditors for reduced payments</li>
          <li><strong>Debt Consolidation:</strong> Single loan to pay off multiple debts</li>
        </ul>

        <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 my-6">
          <p className="text-sm">
            <strong>Important:</strong> This calculator provides an estimate only. Lenders use more comprehensive 
            affordability assessments that include living expenses, dependents, and other factors. Always consult 
            with a financial advisor or credit provider for accurate lending decisions.
          </p>
        </div>
      </div>
    </div>
  );
}
