'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';
import RelatedCalculators from '@/components/calculators/RelatedCalculators';

type Mode = 'simple' | 'compound';
type Frequency = 'annual' | 'semiannual' | 'quarterly' | 'monthly' | 'weekly' | 'daily';

interface YearlyData {
  year: number;
  amount: number;
}

export default function InterestCalculator() {
  const [mode, setMode] = useState<Mode>('compound');
  const [principal, setPrincipal] = useState<string>('10000');
  const [rate, setRate] = useState<string>('5');
  const [years, setYears] = useState<string>('10');
  const [frequency, setFrequency] = useState<Frequency>('annual');
  const [monthlyContribution, setMonthlyContribution] = useState<string>('0');
  
  const [totalAmount, setTotalAmount] = useState<number>(0);
  const [totalInterest, setTotalInterest] = useState<number>(0);
  const [schedule, setSchedule] = useState<YearlyData[]>([]);

  const formatCurrency = (value: number): string => {
    return `R ${value.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
  };

  const getCompoundingFrequency = (freq: Frequency): number => {
    switch (freq) {
      case 'annual': return 1;
      case 'semiannual': return 2;
      case 'quarterly': return 4;
      case 'monthly': return 12;
      case 'weekly': return 52;
      case 'daily': return 365;
      default: return 1;
    }
  };

  useEffect(() => {
    const p = parseFloat(principal) || 0;
    const r = (parseFloat(rate) || 0) / 100;
    const t = parseFloat(years) || 0;
    const contribution = parseFloat(monthlyContribution) || 0;
    
    if (p <= 0 || r < 0 || t <= 0) {
      setTotalAmount(0);
      setTotalInterest(0);
      setSchedule([]);
      return;
    }

    let finalAmount = 0;
    const yearlySchedule: YearlyData[] = [];

    if (mode === 'simple') {
      // Simple Interest: A = P(1 + rt)
      finalAmount = p * (1 + r * t);
      
      // Generate yearly breakdown
      for (let year = 1; year <= t; year++) {
        const amount = p * (1 + r * year);
        yearlySchedule.push({ year, amount });
      }
    } else {
      // Compound Interest with monthly contributions
      const n = getCompoundingFrequency(frequency);
      
      // Calculate compound interest on principal
      const principalGrowth = p * Math.pow(1 + r / n, n * t);
      
      // Calculate future value of monthly contributions (annuity)
      let contributionGrowth = 0;
      if (contribution > 0) {
        const monthlyRate = r / 12;
        const months = t * 12;
        contributionGrowth = contribution * ((Math.pow(1 + monthlyRate, months) - 1) / monthlyRate);
      }
      
      finalAmount = principalGrowth + contributionGrowth;
      
      // Generate yearly breakdown
      for (let year = 1; year <= t; year++) {
        const yearPrincipal = p * Math.pow(1 + r / n, n * year);
        let yearContributions = 0;
        if (contribution > 0) {
          const monthlyRate = r / 12;
          const months = year * 12;
          yearContributions = contribution * ((Math.pow(1 + monthlyRate, months) - 1) / monthlyRate);
        }
        yearlySchedule.push({ year, amount: yearPrincipal + yearContributions });
      }
    }

    const totalInvested = p + (contribution * 12 * t);
    const interest = finalAmount - totalInvested;

    setTotalAmount(finalAmount);
    setTotalInterest(interest);
    setSchedule(yearlySchedule);
  }, [principal, rate, years, mode, frequency, monthlyContribution]);

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Interest Calculator</h1>
        <p className="mt-2 text-gray-600">
          Calculate simple or compound interest on your investments and savings.
        </p>
      </div>

      {/* Top Banner Ad */}
      <LeaderboardAd />

      {/* Calculator Tool */}
      <div className="bg-white rounded-xl shadow-md p-6">
        {/* Mode Toggle */}
        <div className="mb-6">
          <label className="block text-sm font-medium text-gray-700 mb-2">Interest Type</label>
          <div className="flex gap-4">
            <button
              onClick={() => setMode('simple')}
              className={`flex-1 py-3 px-4 rounded-lg font-medium transition-colors ${
                mode === 'simple'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              Simple Interest
            </button>
            <button
              onClick={() => setMode('compound')}
              className={`flex-1 py-3 px-4 rounded-lg font-medium transition-colors ${
                mode === 'compound'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              Compound Interest
            </button>
          </div>
        </div>

        <div className="space-y-4">
          {/* Principal Amount */}
          <div>
            <label htmlFor="principal" className="block text-sm font-medium text-gray-700 mb-1">
              Principal Amount (R)
            </label>
            <input
              type="number"
              id="principal"
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={principal}
              onChange={(e) => setPrincipal(e.target.value)}
              min="0"
              step="1000"
            />
          </div>

          {/* Interest Rate */}
          <div>
            <label htmlFor="rate" className="block text-sm font-medium text-gray-700 mb-1">
              Annual Interest Rate (%)
            </label>
            <input
              type="number"
              id="rate"
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={rate}
              onChange={(e) => setRate(e.target.value)}
              min="0"
              step="0.1"
            />
          </div>

          {/* Time Period */}
          <div>
            <label htmlFor="years" className="block text-sm font-medium text-gray-700 mb-1">
              Time Period (Years)
            </label>
            <input
              type="number"
              id="years"
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={years}
              onChange={(e) => setYears(e.target.value)}
              min="1"
              step="1"
            />
          </div>

          {/* Compounding Frequency (only for compound interest) */}
          {mode === 'compound' && (
            <div>
              <label htmlFor="frequency" className="block text-sm font-medium text-gray-700 mb-1">
                Compounding Frequency
              </label>
              <select
                id="frequency"
                className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                value={frequency}
                onChange={(e) => setFrequency(e.target.value as Frequency)}
              >
                <option value="annual">Annually</option>
                <option value="semiannual">Semi-Annually</option>
                <option value="quarterly">Quarterly</option>
                <option value="monthly">Monthly</option>
                <option value="weekly">Weekly</option>
                <option value="daily">Daily</option>
              </select>
            </div>
          )}

          {/* Monthly Contribution */}
          <div>
            <label htmlFor="contribution" className="block text-sm font-medium text-gray-700 mb-1">
              Monthly Contribution (R) - Optional
            </label>
            <input
              type="number"
              id="contribution"
              className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              value={monthlyContribution}
              onChange={(e) => setMonthlyContribution(e.target.value)}
              min="0"
              step="100"
            />
          </div>
        </div>

        {/* Results */}
        <div className="mt-8 bg-blue-50 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Results</h3>
          <div className="space-y-3">
            <div className="flex justify-between items-center">
              <span className="text-gray-700">Total Amount:</span>
              <span className="text-2xl font-bold text-blue-600">{formatCurrency(totalAmount)}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-700">Total Interest Earned:</span>
              <span className="text-xl font-semibold text-green-600">{formatCurrency(totalInterest)}</span>
            </div>
            <div className="flex justify-between items-center text-sm">
              <span className="text-gray-600">Principal + Contributions:</span>
              <span className="font-medium text-gray-900">
                {formatCurrency(parseFloat(principal) + (parseFloat(monthlyContribution) * 12 * parseFloat(years)))}
              </span>
            </div>
          </div>
        </div>

        {/* Yearly Breakdown */}
        {schedule.length > 0 && (
          <div className="mt-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-3">Yearly Breakdown</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
              {schedule.slice(0, 12).map((row) => (
                <div key={row.year} className="bg-white border border-gray-200 rounded p-3 text-sm flex items-center justify-between">
                  <span>Year {row.year}</span>
                  <span className="font-medium">{formatCurrency(row.amount)}</span>
                </div>
              ))}
            </div>
            {schedule.length > 12 && (
              <p className="text-sm text-gray-500 mt-2">Showing first 12 years. Increase years to see more impact.</p>
            )}
          </div>
        )}
      </div>

      <InContentAd />

      {/* Related Calculators */}
      <RelatedCalculators currentCalculatorId="interest-calculator" category="Investing" />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Interest</h2>
        <p>
          <strong>Simple interest</strong> is calculated only on the principal amount: A = P(1 + rt). 
          <strong>Compound interest</strong> grows on both principal and accumulated interest: A = P(1 + r/n)<sup>nt</sup>.
        </p>
        
        <h3>Simple vs Compound Interest</h3>
        <p>
          Simple interest is straightforward - you earn interest only on your initial investment. 
          Compound interest, however, allows you to earn interest on your interest, leading to exponential growth over time.
        </p>

        <h3>FAQ</h3>
        <div className="space-y-4">
          <div>
            <h4 className="font-semibold">Which frequency should I choose?</h4>
            <p>More frequent compounding (e.g., monthly or daily) yields slightly higher returns than annual compounding, all else equal.</p>
          </div>
          <div>
            <h4 className="font-semibold">How do monthly contributions work?</h4>
            <p>Monthly contributions are added to your investment each month and also earn interest, significantly boosting your final amount.</p>
          </div>
          <div>
            <h4 className="font-semibold">Are contributions pre- or post-tax?</h4>
            <p>This tool assumes nominal contributions. Consult your plan details for specific tax treatment.</p>
          </div>
        </div>
      </div>
    </div>
  );
}
