'use client';

import { useState, useEffect } from 'react';
import { LeaderboardAd, InContentAd } from '@/components/ads/AdBanner';

export default function InflationCalculator() {
  const [amount, setAmount] = useState<string>('10000');
  const [startYear, setStartYear] = useState<string>('2020');
  const [endYear, setEndYear] = useState<string>('2025');
  const [inflationRate, setInflationRate] = useState<string>('5.5');
  
  const [futureValue, setFutureValue] = useState<number>(0);
  const [purchasingPowerLoss, setPurchasingPowerLoss] = useState<number>(0);
  const [totalInflation, setTotalInflation] = useState<number>(0);
  const [yearlyBreakdown, setYearlyBreakdown] = useState<any[]>([]);

  useEffect(() => {
    calculateInflation();
  }, [amount, startYear, endYear, inflationRate]);

  const calculateInflation = () => {
    const principal = parseFloat(amount) || 0;
    const rate = parseFloat(inflationRate) / 100 || 0;
    const start = parseInt(startYear) || new Date().getFullYear();
    const end = parseInt(endYear) || new Date().getFullYear();
    const years = end - start;

    if (principal <= 0 || years <= 0 || rate < 0) {
      resetResults();
      return;
    }

    // Calculate future value adjusted for inflation
    const future = principal * Math.pow(1 + rate, years);
    const loss = future - principal;
    const totalInflationPercent = ((future - principal) / principal) * 100;

    // Generate yearly breakdown
    const breakdown = [];
    let currentValue = principal;
    
    for (let i = 0; i <= years; i++) {
      const year = start + i;
      const yearInflation = currentValue * rate;
      const valueAfterInflation = currentValue + yearInflation;
      const purchasingPower = (principal / valueAfterInflation) * 100;
      
      breakdown.push({
        year,
        startValue: currentValue,
        inflationAmount: yearInflation,
        endValue: valueAfterInflation,
        purchasingPower: purchasingPower,
        cumulativeInflation: ((valueAfterInflation - principal) / principal) * 100
      });
      
      currentValue = valueAfterInflation;
    }

    setFutureValue(future);
    setPurchasingPowerLoss(loss);
    setTotalInflation(totalInflationPercent);
    setYearlyBreakdown(breakdown);
  };

  const resetResults = () => {
    setFutureValue(0);
    setPurchasingPowerLoss(0);
    setTotalInflation(0);
    setYearlyBreakdown([]);
  };

  const formatCurrency = (amount: number): string => {
    return new Intl.NumberFormat('en-ZA', {
      style: 'currency',
      currency: 'ZAR',
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  const currentYear = new Date().getFullYear();
  const years = parseInt(endYear) - parseInt(startYear);

  return (
    <div className="space-y-8">
      {/* Page Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900">Inflation Calculator</h1>
        <p className="mt-2 text-gray-600">
          See how inflation affects the purchasing power of your money over time in South Africa.
        </p>
      </div>

      <LeaderboardAd />

      {/* Calculator */}
      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Left Column - Inputs */}
          <div className="lg:col-span-2 space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Amount */}
              <div>
                <label htmlFor="amount" className="block text-sm font-medium text-gray-700 mb-1">
                  Amount (R)
                </label>
                <input
                  type="text"
                  id="amount"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={amount}
                  onChange={(e) => setAmount(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="10000"
                />
              </div>

              {/* Start Year */}
              <div>
                <label htmlFor="start-year" className="block text-sm font-medium text-gray-700 mb-1">
                  Start Year
                </label>
                <input
                  type="number"
                  id="start-year"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={startYear}
                  onChange={(e) => setStartYear(e.target.value)}
                  min="1900"
                  max={currentYear}
                />
              </div>

              {/* End Year */}
              <div>
                <label htmlFor="end-year" className="block text-sm font-medium text-gray-700 mb-1">
                  End Year
                </label>
                <input
                  type="number"
                  id="end-year"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={endYear}
                  onChange={(e) => setEndYear(e.target.value)}
                  min={parseInt(startYear) + 1}
                  max={currentYear + 50}
                />
              </div>

              {/* Inflation Rate */}
              <div>
                <label htmlFor="inflation-rate" className="block text-sm font-medium text-gray-700 mb-1">
                  Annual Inflation Rate (%)
                </label>
                <input
                  type="text"
                  id="inflation-rate"
                  className="focus:ring-blue-500 focus:border-blue-500 block w-full px-3 py-2 text-sm border-gray-300 rounded-md border"
                  value={inflationRate}
                  onChange={(e) => setInflationRate(e.target.value.replace(/[^0-9.]/g, ''))}
                  placeholder="5.5"
                />
                <p className="mt-1 text-xs text-gray-500">SA average: 4-6%</p>
              </div>
            </div>
          </div>

          {/* Right Column - Results */}
          <div className="space-y-4">
            <div className="bg-gradient-to-br from-red-50 to-orange-100 rounded-lg p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Impact Summary</h3>
              
              <div className="space-y-3">
                <div>
                  <p className="text-xs text-gray-600">Original Amount</p>
                  <p className="text-xl font-bold text-gray-900">{formatCurrency(parseFloat(amount) || 0)}</p>
                </div>

                <div className="border-t border-orange-200 pt-3">
                  <p className="text-xs text-gray-600">Future Value Needed</p>
                  <p className="text-xl font-bold text-orange-600">{formatCurrency(futureValue)}</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Purchasing Power Loss</p>
                  <p className="text-lg font-semibold text-red-600">{formatCurrency(purchasingPowerLoss)}</p>
                </div>

                <div className="border-t border-orange-200 pt-3">
                  <p className="text-xs text-gray-600">Total Inflation</p>
                  <p className="text-lg font-bold text-gray-900">{totalInflation.toFixed(2)}%</p>
                </div>

                <div>
                  <p className="text-xs text-gray-600">Time Period</p>
                  <p className="text-sm font-medium text-gray-900">{years} years</p>
                </div>
              </div>
            </div>

            <div className="bg-gray-50 rounded-lg p-4 text-sm">
              <p className="text-gray-700">
                <strong>What this means:</strong> In {years} years, you'll need {formatCurrency(futureValue)} to buy what {formatCurrency(parseFloat(amount) || 0)} buys today.
              </p>
            </div>
          </div>
        </div>

        {/* Yearly Breakdown */}
        {yearlyBreakdown.length > 0 && (
          <div className="mt-8">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Year-by-Year Breakdown</h3>
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Year</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Start Value</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Inflation</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">End Value</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Purchasing Power</th>
                    <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Cumulative %</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {yearlyBreakdown.map((row, index) => (
                    <tr key={index} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                      <td className="px-4 py-3 text-sm text-gray-900">{row.year}</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{formatCurrency(row.startValue)}</td>
                      <td className="px-4 py-3 text-sm text-right text-red-600">{formatCurrency(row.inflationAmount)}</td>
                      <td className="px-4 py-3 text-sm text-right font-medium text-gray-900">{formatCurrency(row.endValue)}</td>
                      <td className="px-4 py-3 text-sm text-right text-orange-600">{row.purchasingPower.toFixed(1)}%</td>
                      <td className="px-4 py-3 text-sm text-right text-gray-900">{row.cumulativeInflation.toFixed(2)}%</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
      </div>

      <InContentAd />

      {/* Information Section */}
      <div className="prose max-w-none">
        <h2>Understanding Inflation in South Africa</h2>
        <p>
          Inflation is the rate at which the general level of prices for goods and services rises, eroding purchasing power. 
          In South Africa, the South African Reserve Bank (SARB) targets an inflation rate of 3-6% per year.
        </p>

        <h2>How to Use the Inflation Calculator</h2>
        <ol>
          <li>Enter the amount of money you want to analyze</li>
          <li>Set the start year (when you have the money)</li>
          <li>Set the end year (when you want to use the money)</li>
          <li>Enter the expected annual inflation rate (SA average is 4-6%)</li>
          <li>View how much purchasing power you'll lose over time</li>
        </ol>

        <h2>South African Inflation Trends</h2>
        <p>
          South Africa's inflation rate has historically ranged between 3% and 7%, with the SARB using monetary policy 
          to keep it within the 3-6% target range. Factors affecting SA inflation include:
        </p>
        <ul>
          <li><strong>Fuel Prices:</strong> Heavily influenced by global oil prices and the Rand exchange rate</li>
          <li><strong>Food Prices:</strong> Affected by weather patterns, droughts, and global commodity prices</li>
          <li><strong>Electricity Costs:</strong> Eskom tariff increases impact overall inflation</li>
          <li><strong>Exchange Rate:</strong> A weaker Rand increases import costs and inflation</li>
          <li><strong>Interest Rates:</strong> SARB adjusts repo rates to control inflation</li>
        </ul>

        <h2>Protecting Your Money from Inflation</h2>
        <ul>
          <li><strong>Invest in Assets:</strong> Stocks, property, and other assets often outpace inflation</li>
          <li><strong>Inflation-Linked Bonds:</strong> Government bonds that adjust for inflation</li>
          <li><strong>Diversify Internationally:</strong> Reduce exposure to Rand depreciation</li>
          <li><strong>Review Regularly:</strong> Adjust your investment strategy as inflation changes</li>
          <li><strong>Consider Real Returns:</strong> Always calculate returns after inflation</li>
        </ul>

        <h2>Historical South African Inflation</h2>
        <p>
          Key periods in SA inflation history:
        </p>
        <ul>
          <li><strong>1980s:</strong> High inflation averaging 14-15% annually</li>
          <li><strong>1990s:</strong> Gradual decline to single digits</li>
          <li><strong>2000-2020:</strong> Generally within 3-7% range</li>
          <li><strong>2020-2023:</strong> Fluctuations due to COVID-19 and global factors</li>
          <li><strong>Current:</strong> SARB maintains 3-6% inflation target</li>
        </ul>

        <h2>Inflation vs Investment Returns</h2>
        <p>
          When planning your finances, always consider "real returns" - the return on investment after accounting for inflation. 
          For example, if your investment earns 8% but inflation is 5%, your real return is only 3%.
        </p>

        <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 my-6">
          <p className="text-sm">
            <strong>Note:</strong> This calculator uses a constant inflation rate for simplicity. Actual inflation varies 
            year by year. For accurate historical calculations, use actual CPI data from Statistics South Africa (Stats SA).
          </p>
        </div>
      </div>
    </div>
  );
}
